/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import org.mule.runtime.api.util.MultiMap;

import java.net.URI;

import io.netty.handler.codec.http.QueryStringEncoder;

/**
 * Utility class for the Netty HTTP Service implementation.
 */
public final class HttpUtils {

  public static final String QUERY_STRING_DELIMITER = "\\?";
  public static final String QUERY_PARAMS_SEPARATOR = "&";
  public static final String QUERY_PARAM_VALUE_DELIMITER = "=";

  private HttpUtils() {
    // private constructor to avoid wrong instantiations
  }

  /**
   * Builds a URI String from a passed {@link URI}, by adding the passed query parameters.
   *
   * @param uri         the uri. It must contain schema and host different from {@code null}.
   * @param queryParams the additional query parameters.
   * @return if {@code queryParams} is null or empty, it just returns the {@code uri}'s ASCII string. Otherwise, it returns an URI
   *         String that adds the passed query params to the original uri.
   *
   * @throws IllegalArgumentException if the URI provided doesn't contain schema or host.
   */
  public static String buildUriString(URI uri, MultiMap<String, String> queryParams) {
    String uriAsString = ensureSchemeAndHost(uri).toASCIIString();
    if (queryParams == null || queryParams.isEmpty()) {
      return uriAsString;
    }

    String[] splitUri = uriAsString.split(QUERY_STRING_DELIMITER);
    QueryStringEncoder encoder = new QueryStringEncoder(splitUri[0]);
    if (splitUri.length > 1) {
      String[] uriQueryParams = splitUri[1].split(QUERY_PARAMS_SEPARATOR);
      for (String param : uriQueryParams) {
        String[] keyValueParam = param.split(QUERY_PARAM_VALUE_DELIMITER);
        encoder.addParam(keyValueParam[0], keyValueParam.length > 1 ? keyValueParam[1] : null);
      }
    }

    for (String key : queryParams.keySet()) {
      for (String value : queryParams.getAll(key)) {
        encoder.addParam(key, value);
      }
    }
    return encoder.toString();
  }

  /**
   * Ensures that an {@link URI} has scheme and host.
   *
   * @param uri an {@link URI}.
   * @return the same {@link URI}.
   * @throws IllegalArgumentException if the passed uri does not contain scheme or host.
   */
  public static URI ensureSchemeAndHost(URI uri) {
    if (uri.getScheme() == null) {
      throw new IllegalArgumentException("The uri provided '" + uri + "' must contain a scheme.");
    }
    if (uri.getHost() == null) {
      throw new IllegalArgumentException("The uri provided '" + uri + "' must contain a host.");
    }
    return uri;
  }
}
