/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.test.netty.AllureConstants.HTTP_2;
import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_DEFERRED_RESPONSES;
import static org.mule.tck.junit4.matcher.Eventually.eventually;
import static org.mule.tck.junit4.matcher.FunctionExpressionMatcher.expressionMatches;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.concurrent.Executors.newSingleThreadScheduledExecutor;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalToCompressingWhiteSpace;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.service.http.test.common.util.AsyncEntityReader;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.tck.junit5.DynamicPort;

import java.io.IOException;
import java.io.Writer;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledExecutorService;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Feature(HTTP_2)
@Story(HTTP_2_DEFERRED_RESPONSES)
class Http2DeferringResponsePayloadTestCase extends AbstractHttpServiceTestCase {

  private static final ScheduledExecutorService executorService = newSingleThreadScheduledExecutor();

  @DynamicPort(systemProperty = "serverPort")
  Integer serverPort;

  private HttpServer httpServer;
  private HttpClient httpClient;

  private final CompletableFuture<Writer> capturedResponseWriter = new CompletableFuture<>();

  public Http2DeferringResponsePayloadTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setUp() throws Exception {
    httpServer = createServer();
    httpClient = createClient();
  }

  @AfterEach
  void tearDown() {
    httpClient.stop();
    httpServer.stop().dispose();
  }

  @Test
  void writeInParts() throws ExecutionException, InterruptedException, IOException {
    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();

    var asyncReader = new AsyncEntityReader(response);
    executorService.submit(asyncReader);

    var writer = capturedResponseWriter.get();
    writer.write("Chunk 1\n");
    writer.write("Chunk 2\n");
    writer.write("Chunk 3\n");
    writer.flush();

    checkPayload(asyncReader, """
        Chunk 1
        Chunk 2
        Chunk 3
        """);
    assertThat(asyncReader.finished(), is(false));

    writer.write("Chunk 4\n");
    writer.flush();
    writer.close();

    checkPayload(asyncReader, """
        Chunk 1
        Chunk 2
        Chunk 3
        Chunk 4
        """);
    assertThat(asyncReader.finished(), is(true));
  }

  private static void checkPayload(AsyncEntityReader asyncReader, String expectedContent) {
    assertThat(asyncReader, eventually(expressionMatches(AsyncEntityReader::partialPayload, equalToCompressingWhiteSpace(
                                                                                                                         expectedContent))));
  }

  private HttpClient createClient() throws CreateException {
    var client = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setName("HTTP/2 Client")
        .setHttp2Config(new Http2ProtocolConfig(true))
        .setTlsContextFactory(TlsContextFactory.builder().trustStorePath("trustStore")
            .trustStorePassword("mulepassword").insecureTrustStore(true).build())
        .build());
    client.start();
    return client;
  }

  private HttpServer createServer()
      throws ServerCreationException, IOException, CreateException {
    var server = service.getServerFactory().create(new HttpServerConfiguration.Builder()
        .setName("HTTP/2 Server")
        .setHost("localhost")
        .setPort(serverPort)
        .setHttp2Config(new Http2ProtocolConfig(true))
        .setTlsContextFactory(TlsContextFactory.builder()
            .keyStorePath("serverKeystore")
            .keyStorePassword("mulepassword").keyAlias("muleserver").keyPassword("mulepassword").keyStoreAlgorithm("PKIX")
            .build())
        .build());

    server.start();
    server.addRequestHandler("/test", (ctx, callback) -> capturedResponseWriter
        .complete(callback.startResponse(HttpResponse.builder().build(), new NoOpResponseStatusCallback(), UTF_8)));

    return server;
  }
}
