/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONNECTION;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Values.CLOSE;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.http.HttpVersion.HTTP_1_0;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;

import java.io.IOException;

import org.apache.commons.io.IOUtils;
import org.apache.http.HttpVersion;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.hamcrest.Matcher;
import org.junit.jupiter.api.Test;

public class HttpServerTransfer10TestCase extends HttpServerTransferTestCase {

  public HttpServerTransfer10TestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  public HttpVersion getVersion() {
    return HTTP_1_0;
  }

  @Test
  void defaultsStreamingWhenEmpty() throws Exception {
    verify10Headers(EMPTY, CLOSE, is(nullValue()), EMPTY);
  }

  @Test
  void defaultsStreamingWhenBytes() throws Exception {
    verify10Headers(BYTES, CLOSE, is(nullValue()), DATA);
  }

  @Test
  void defaultsStreamingWhenMultipart() throws Exception {
    verify10Headers(MULTIPART, CLOSE, is(nullValue()), MULTIPART_DATA);
  }

  @Test
  void defaultsStreamingWhenStream() throws Exception {
    verify10Headers(STREAM, CLOSE, is(nullValue()), DATA);
  }

  private void verify10Headers(String path, String expectedConnection, Matcher<Object> contentLengthMatcher,
                               String expectedBody)
      throws IOException {
    try (CloseableHttpClient httpClient = HttpClients.createDefault()) {
      HttpGet httpGet = new HttpGet(getUri(path));
      httpGet.setProtocolVersion(getVersion());
      try (CloseableHttpResponse response = httpClient.execute(httpGet)) {
        assertThat(getHeaderValue(response, CONNECTION), is(expectedConnection));
        assertThat(getHeaderValue(response, CONTENT_LENGTH), contentLengthMatcher);
        assertThat(IOUtils.toString(response.getEntity().getContent(), UTF_8), is(expectedBody));
      }
    }
  }

}
