/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message.content;

import static org.mule.service.http.netty.impl.server.util.HttpParser.parseMultipartContent;

import static java.util.Optional.ofNullable;

import static org.apache.commons.io.IOUtils.toByteArray;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Optional;

/**
 * Represents a received multipart body, which can be accessed raw or parsed.
 *
 * @since 1.0
 */
public class StreamedMultipartHttpEntity implements HttpEntity {

  private InputStream content;
  private String contentType;
  private Long contentLength;

  public StreamedMultipartHttpEntity(InputStream content, String contentType) {
    this.content = content;
    this.contentType = contentType;
  }

  public StreamedMultipartHttpEntity(InputStream content, String contentType, Long contentLength) {
    this(content, contentType);
    this.contentLength = contentLength;
  }

  @Override
  public boolean isStreaming() {
    return true;
  }

  @Override
  public boolean isComposed() {
    return true;
  }

  @Override
  public InputStream getContent() throws UnsupportedOperationException {
    return content;
  }

  @Override
  public byte[] getBytes() throws UnsupportedOperationException {
    try {
      return toByteArray(content);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public Collection<HttpPart> getParts() throws IOException, UnsupportedOperationException {
    return parseMultipartContent(content, contentType);
  }

  @Override
  public Optional<Long> getLength() {
    return ofNullable(contentLength);
  }

}
