/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.service.http.netty.impl.util.MuleToNettyUtils.adaptResponseWithoutBody;

import org.mule.runtime.http.api.domain.HttpProtocol;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.streaming.StatusCallback;

import java.io.IOException;
import java.io.InputStream;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.DefaultLastHttpContent;

/**
 * Basic implementation of {@link ResponseSender}, that implements {@link #send()} as a template method. First, it saves the
 * header part of the response, and then delegates the content sending to the child implementations by calling
 * {@link #sendContent()}. The implementations of {@link #sendContent()} may call the protected methods of this class in order to
 * simplify their logic. When calling the sending methods, they will actually send the saved header part of the response if it
 * wasn't done yet. It's done this way so that if an exception is raised during the reading of the response content (within the
 * implementations of {@link #sendContent()}), we can send a corresponding failure response instead. Otherwise, if we send it as
 * soon as possible, we can't change the response's status code.
 */
public abstract class BaseResponseSender implements ResponseSender {

  protected final HttpRequest request;
  protected final HttpResponse response;

  private final ChannelHandlerContext ctx;
  private final StatusCallback statusCallback;
  private io.netty.handler.codec.http.HttpResponse savedHeader;

  protected BaseResponseSender(HttpRequest request, ChannelHandlerContext ctx, HttpResponse response,
                               StatusCallback statusCallback) {
    checkArgument(statusCallback != null, "statusCallback can't be null");

    this.request = request;
    this.response = response;
    this.ctx = ctx;
    this.statusCallback = statusCallback;
    this.savedHeader = null;
  }

  @Override
  public void send() throws IOException {
    // Just save the header part of the response to send later. It's done this way so that if an exception is raised
    // during the reading of the response CONTENT, we can send a corresponding failure response instead. Otherwise, if
    // we send it at this point, we can't change the response's status code.
    saveHeader(getNettyResponseHeader());
    sendContent();
  }

  private void saveHeader(io.netty.handler.codec.http.HttpResponse nettyResponse) {
    this.savedHeader = nettyResponse;
  }

  protected void sendHeaderIfNeeded() {
    if (savedHeader != null) {
      ctx.write(savedHeader);
      savedHeader = null;
    }
  }

  private io.netty.handler.codec.http.HttpResponse getNettyResponseHeader() {
    HttpProtocol httpProtocol = request.getProtocol();
    return adaptResponseWithoutBody(response, httpProtocol);
  }

  protected ChannelPromise finishStreamingPromise(InputStream stream) {
    return createPromise(new FinishStreamingListener(stream, statusCallback));
  }

  protected ByteBuf createBuffer(int size) {
    return ctx.alloc().buffer(size, size);
  }

  protected void sendLastContentAndFinishStreaming(ByteBuf content, InputStream contentAsInputStream) {
    sendHeaderIfNeeded();
    ctx.writeAndFlush(new DefaultLastHttpContent(content), finishStreamingPromise(contentAsInputStream));
  }

  protected ChannelPromise createPromise(ChannelFutureListener futureListener) {
    ChannelPromise channelPromise = ctx.newPromise();
    channelPromise.addListener(futureListener);
    return channelPromise;
  }

  protected abstract void sendContent() throws IOException;
}
