/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.client.proxy;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.service.http.netty.impl.util.SslContextHelper.sslContextForClient;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.fail;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.netty.utils.HttpProxyRule;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.server.TestHttpServer;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.util.concurrent.TimeoutException;

import io.netty.handler.ssl.SslContext;
import io.qameta.allure.Issue;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

@Issue("W-16606326")
public abstract class AbstractSimpleProxyTestCase extends AbstractMuleTestCase {

  private static final String RESPONSE_PAYLOAD = "Hello from target server!";

  @Rule
  public DynamicPort proxyPort = new DynamicPort("proxyPort");

  @Rule
  public DynamicPort targetPort = new DynamicPort("targetPort");

  @Rule
  public TestHttpServer targetServer;

  @Rule
  public HttpProxyRule proxy;

  private final boolean useSsl;
  private SslContext sslContext;

  protected AbstractSimpleProxyTestCase(boolean useSsl) {
    this.useSsl = useSsl;
    this.targetServer = new TestHttpServer("localhost", targetPort.getNumber(), useSsl);
    this.proxy = new HttpProxyRule(proxyPort.getNumber(), targetPort.getNumber(), useSsl);
  }

  @Before
  public void setup() {
    targetServer.addRequestHandler("/hello", (request, responseSender) -> {
      HttpEntity helloContent = new StringHttpEntity(RESPONSE_PAYLOAD);
      responseSender.responseReady(new ResponseWithoutHeaders(OK, helloContent), new NoOpResponseStatusCallback());
    }).start();

    targetServer.addRequestHandler("/hello_and_check_headers", (request, responseSender) -> {
      String proxyAuthorizationHeader = request.getRequest().getHeaderValue("Proxy-Authorization");
      if (proxyAuthorizationHeader != null) {
        fail("Proxy-Authorization header should not be present");
      }
      HttpEntity helloContent = new StringHttpEntity(RESPONSE_PAYLOAD);
      responseSender.responseReady(new ResponseWithoutHeaders(OK, helloContent), new NoOpResponseStatusCallback());
    }).start();

    sslContext = getSslContextIfNeeded(useSsl);
  }

  private SslContext getSslContextIfNeeded(boolean useSsl) {
    if (!useSsl) {
      return null;
    }

    TlsContextFactory tlsContextFactory = null;
    try {
      tlsContextFactory = TlsContextFactory.builder()
          .trustStorePath("trustStore")
          .trustStorePassword("mulepassword")
          .keyStorePath("clientKeystore")
          .keyStorePassword("mulepassword")
          .keyPassword("mulepassword")
          .build();
    } catch (CreateException e) {
      throw new RuntimeException(e);
    }

    return sslContextForClient(tlsContextFactory, true, false);
  }

  @Test
  public void sendThroughProxyBasicAuth() throws IOException, TimeoutException, CreateException {
    ProxyConfig proxyConfig = ProxyConfig.builder()
        .host("localhost").port(proxyPort.getNumber())
        .username("Eze").password("NotMyActualPassword")
        .build();

    testSendThroughProxy(proxyConfig, "hello");
  }

  @Test
  public void sendThroughProxyNoAuth() throws IOException, TimeoutException, CreateException {
    ProxyConfig proxyConfig = ProxyConfig.builder()
        .host("localhost").port(proxyPort.getNumber())
        .build();

    testSendThroughProxy(proxyConfig, "hello");
  }

  @Test
  public void sendThroughProxyNtlmAuth() throws IOException, TimeoutException, CreateException {
    ProxyConfig.NtlmProxyConfig proxyConfig = ProxyConfig.NtlmProxyConfig.builder()
        .host("localhost").port(proxyPort.getNumber())
        .username("Eze").password("NotMyActualPassword")
        .ntlmDomain("theNtlmDomain")
        .build();

    testSendThroughProxy(proxyConfig, "hello");
  }

  protected void testSendThroughProxy(ProxyConfig proxyConfig, String url) throws IOException, TimeoutException, CreateException {
    NettyHttpClient client = NettyHttpClient.builder()
        .withSslContext(sslContext)
        .withProxyConfig(proxyConfig)
        .build();
    client.start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("%%s://localhost:%%d/%s".formatted(url), useSsl ? "https" : "http", targetPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();

    try {
      HttpResponse response = client.send(httpRequest);
      assertThat(response.getStatusCode(), is(200));
      assertThat(new String(response.getEntity().getBytes()), is(RESPONSE_PAYLOAD));
      assertThat("The request has to pass through the proxy server", proxy.hasConnections(), is(true));
    } finally {
      client.stop();
    }
  }
}
