/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.util;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;
import static org.mule.service.http.netty.impl.util.HttpUtils.buildUriString;

import static java.net.URI.create;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThrows;

import org.mule.runtime.api.util.MultiMap;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.net.URI;

import io.qameta.allure.Issue;
import org.junit.Test;

public class HttpUtilsTestCase extends AbstractMuleTestCase {

  @Test
  @Issue("W-16010240")
  public void tryGeneratingUriStringWithoutSchemeFails() {
    IllegalArgumentException exception = assertThrows(IllegalArgumentException.class, () -> {
      buildUriString(create("localhost"), emptyMultiMap());
    });
    assertThat(exception.getMessage(), is("The uri provided 'localhost' must contain a scheme."));
  }

  @Test
  @Issue("W-16010240")
  public void tryGeneratingUriStringWithoutHostFails() {
    IllegalArgumentException exception = assertThrows(IllegalArgumentException.class, () -> {
      buildUriString(create("http:/"), emptyMultiMap());
    });
    assertThat(exception.getMessage(), is("The uri provided 'http:/' must contain a host."));
  }

  @Test
  public void generateUriStringWithExistingAndExtraQueryParams() {
    MultiMap<String, String> params = new MultiMap<>();
    params.put("other", "value");
    URI uri = create("http://localhost:8080/myresource?qpname=qpvalue");
    String result = buildUriString(uri, params);
    assertThat(result, is("http://localhost:8080/myresource?qpname=qpvalue&other=value"));
  }

  @Test
  public void generateUriStringWithEmptyQueryParams() {
    URI uri = create("http://localhost:8080/myresource?qpname=qpvalue");
    String result = buildUriString(uri, emptyMultiMap());
    assertThat(result, is("http://localhost:8080/myresource?qpname=qpvalue"));
  }

  @Test
  public void generateUriStringWithNullQueryParams() {
    URI uri = create("http://localhost:8080/myresource?qpname=qpvalue");
    String result = buildUriString(uri, null);
    assertThat(result, is("http://localhost:8080/myresource?qpname=qpvalue"));
  }

  @Test
  @Issue("W-15631603")
  public void testUrlWithPathAndQuery() {
    MultiMap<String, String> params = new MultiMap<>();
    params.put("other", "value");
    URI uri = create("http://localhost:8080/test?q=1");
    String result = buildUriString(uri, params);
    assertThat(result, is("http://localhost:8080/test?q=1&other=value"));
  }

  @Test
  @Issue("W-15631603")
  public void testUrlFragmentWithTrickAuthority() {
    URI uri = create("http://1.2.3.4:81#@5.6.7.8:82/aaa/b?q=xxx");
    String result = buildUriString(uri, emptyMultiMap());
    assertThat(result, is("http://1.2.3.4:81#@5.6.7.8:82/aaa/b?q=xxx"));
  }

  @Test
  @Issue("W-15631603")
  public void testRootWithQueryParameters() {
    URI uri = create("http://localhost:8080/");
    MultiMap<String, String> params = new MultiMap<>();
    params.put("newParam", "newValue");
    String result = buildUriString(uri, params);
    assertThat(result, is("http://localhost:8080/?newParam=newValue"));
  }

  @Test
  @Issue("W-15631603")
  public void testRelativeUrlWithQuery() {
    URI uri = create("http://localhost:8080/test");
    MultiMap<String, String> params = new MultiMap<>();
    params.put("q", "2");
    String result = buildUriString(uri, params);
    assertThat(result, is("http://localhost:8080/test?q=2"));
  }


  // According to RFC 3986 documentation (https://datatracker.ietf.org/doc/html/rfc3986#section-3.5), characters like ? and & in
  // the fragment
  // (after #) should not be treated as part of the query string. The fragment is handled separately.
  @Test
  @Issue("W-15631603")
  public void testUrlWithEmptyAnchor() {
    URI uri = URI.create("http://localhost:8080/resource#");
    String result = buildUriString(uri, emptyMultiMap());
    assertThat(uri.getFragment(), is(""));
    assertThat(result, is("http://localhost:8080/resource#"));
  }

  @Test
  @Issue("W-15631603")
  public void testAnchorWithOneSpecialCharacterAndQueryParameter() {
    URI uri = URI.create("http://localhost:8080/resource#section?param=value&otherParam=otherValue");
    MultiMap<String, String> params = new MultiMap<>();
    params.put("myParam", "myValue");
    String result = buildUriString(uri, params);
    assertThat(uri.getFragment(), is("section?param=value&otherParam=otherValue"));
    assertThat(result, is("http://localhost:8080/resource#section?param=value&otherParam=otherValue&myParam=myValue"));
  }

  @Test
  @Issue("W-15631603")
  public void testAnchorWithMultipleSpecialCharactersAndMultipleQueryParameters() {
    URI uri = URI.create("http://localhost:8080/resource?queryParam=value#section?param1=value1&param2=value2&param3=value3");
    MultiMap<String, String> params = new MultiMap<>();
    params.put("myParam-1", "myValue-1");
    params.put("myParam-2", "myValue-2");
    String result = buildUriString(uri, params);
    assertThat(uri.getFragment(), is("section?param1=value1&param2=value2&param3=value3"));
    assertThat(result, is("http://localhost:8080/resource?queryParam=value%23section&myParam-1=myValue-1&myParam-2=myValue-2"));
  }
}
