/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message;

import static org.mule.runtime.api.metadata.MediaType.MULTIPART_MIXED;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.service.http.netty.impl.message.content.StreamedMultipartHttpEntity;

import java.io.InputStream;
import java.net.InetSocketAddress;

public class NettyHttpRequestAdapter extends NettyHttpMessage implements HttpRequest {

  private static final String PROTOCOL = "http";

  private final InputStream requestContent;
  private HttpEntity body;

  public NettyHttpRequestAdapter(io.netty.handler.codec.http.HttpRequest nettyRequest,
                                 InetSocketAddress localAddress,
                                 InputStream content) {
    super(nettyRequest, localAddress);
    requestContent = content;
  }

  @Override
  protected String getBaseProtocol() {
    return PROTOCOL;
  }

  @Override
  public HttpEntity getEntity() {
    if (this.body == null) {
      final String contentTypeValue = getHeaderValue(CONTENT_TYPE);
      if (contentTypeValue != null && contentTypeValue.contains(MULTIPART_MIXED.getPrimaryType())) {
        if (contentLength >= 0) {
          this.body = new StreamedMultipartHttpEntity(requestContent, contentTypeValue, contentLength);
        } else {
          this.body = new StreamedMultipartHttpEntity(requestContent, contentTypeValue);
        }
      } else {
        if (contentLength > 0) {
          this.body = new InputStreamHttpEntity(requestContent, contentLength);
        } else if (contentLength == 0) {
          this.body = new EmptyHttpEntity();
        } else {
          this.body = new InputStreamHttpEntity(requestContent);
        }
      }
    }
    return this.body;
  }

  public boolean containsHeader(String headerName, String headerValue) {
    if (!containsHeader(headerName)) {
      return false;
    }

    for (String connectionHeader : getHeaderValues(headerName)) {
      if (headerValue.equalsIgnoreCase(connectionHeader)) {
        return true;
      }
    }

    return false;
  }
}
