/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.util;

import static java.nio.charset.StandardCharsets.UTF_8;

import org.mule.runtime.http.api.domain.entity.HttpEntity;

import java.io.IOException;

import org.apache.commons.io.IOUtils;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;

public final class HttpEntityMatcher extends TypeSafeMatcher<HttpEntity> {

  public static Matcher<HttpEntity> hasContent(Matcher<String> contentStringMatcher) {
    return new HttpEntityMatcher(contentStringMatcher);
  }

  private final Matcher<String> contentStringMatcher;

  private HttpEntityMatcher(Matcher<String> contentStringMatcher) {
    this.contentStringMatcher = contentStringMatcher;
  }

  @Override
  protected boolean matchesSafely(HttpEntity item) {
    String contentAsString;
    try {
      contentAsString = IOUtils.toString(item.getContent(), UTF_8);
    } catch (IOException e) {
      return false;
    }
    return contentStringMatcher.matches(contentAsString);
  }

  @Override
  public void describeTo(Description description) {
    description.appendText("A content matching: ");
    contentStringMatcher.describeTo(description);
  }

  @Override
  protected void describeMismatchSafely(HttpEntity item, Description mismatchDescription) {
    String contentAsString;
    try {
      contentAsString = IOUtils.toString(item.getContent(), UTF_8);
      mismatchDescription.appendText("Content was: '").appendText(contentAsString).appendText("'");
    } catch (IOException e) {
      mismatchDescription.appendText(e.toString());
    }
  }
}
