/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.client;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThrows;
import static org.junit.internal.matchers.ThrowableCauseMatcher.hasCause;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.server.HardcodedResponseTcpServer;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junitpioneer.jupiter.Issue;

public class ClientResponseStreamingTestCase extends AbstractMuleTestCase {

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public HardcodedResponseTcpServer server = new HardcodedResponseTcpServer(serverPort.getNumber());

  private HttpClient client;

  @Before
  public void setUp() throws Exception {
    server.setResponse("HTTP/1.1 200 OK\ntransfer-encoding: chunked\n\n1\nA");
    server.setCloseOutputAfterResponse(true);

    client =
        NettyHttpClient.builder().withIOTasksScheduler(executorRule.getExecutor()).withResponseStreamingEnabled(true).build();
    client.start();
  }

  @After
  public void tearDown() {
    if (client != null) {
      client.stop();
    }
  }

  @Test
  public void propagateErrorDuringResponseStreamingToReader() throws IOException, TimeoutException {
    HttpRequest httpRequest = HttpRequest.builder()
        .method("GET")
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .entity(new EmptyHttpEntity()).build();

    HttpResponse response = client.send(httpRequest);
    HttpEntity entity = response.getEntity();
    assertThat("Response entity must be streaming", entity.isStreaming(), is(true));

    InputStream inputStream = entity.getContent();
    IOException exception = assertThrows(IOException.class, () -> {
      boolean eosReached = false;
      while (!eosReached) {
        eosReached = (inputStream.read() == -1);
      }
    });

    assertThat(exception, hasCause(allOf(
                                         instanceOf(IOException.class),
                                         hasMessage(containsString("Remotely closed")))));
  }

  @Test
  @Issue("W-19679238")
  public void allSchedulersBusyDuringResponseCompletionThenFallbackToSameThread()
      throws IOException, InterruptedException, ExecutionException {
    server.setResponse("HTTP/1.1 200 OK\ntransfer-encoding: chunked\r\n\r\n1\r\nA\r\n0\r\n\r\n");
    server.setCloseOutputAfterResponse(false);
    HttpRequest httpRequest = HttpRequest.builder()
        .method("GET")
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .entity(new EmptyHttpEntity()).build();

    HttpEntity entity;
    server.setOnRequestReceived(() -> executorRule.setRejectNext());

    CompletableFuture<HttpResponse> response = client.sendAsync(httpRequest);

    // We force a rejection on the next submission, simulating the case in which the scheduler is fully busy
    executorRule.setRejectNext();

    // At this point there will be a rejected execution when submitting the completion of the response future
    // With the fix, it will be caught and executed in the same thread as a fallback
    entity = response.get().getEntity();
    assertThat("Response entity must be streaming", entity.isStreaming(), is(true));

    InputStream inputStream = entity.getContent();
    boolean eosReached = false;
    while (!eosReached) {
      eosReached = (inputStream.read() == -1);
    }
  }
}
