/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.message;

import static io.netty.handler.codec.http.HttpMethod.GET;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;

import org.mule.runtime.api.util.MultiMap;
import org.mule.service.http.netty.impl.message.NettyHttpRequestAdapter;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.InetSocketAddress;

import io.netty.handler.codec.http.DefaultHttpRequest;
import io.qameta.allure.Issue;
import org.junit.Test;

@Issue("W-15818350")
public class NettyHttpRequestAdapterQueryParamsTestCase extends AbstractMuleTestCase {

  private final InetSocketAddress address = new InetSocketAddress("localhost", 80);

  private final InputStream testPayload = new ByteArrayInputStream("Hello world".getBytes());

  @Test
  public void oneKeyValueQueryParameter() {
    io.netty.handler.codec.http.HttpRequest nettyRequest = withUri("https://test.org/path?key=value");
    org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest = adaptToMule(nettyRequest);

    MultiMap<String, String> queryParams = muleRequest.getQueryParams();
    assertThat(queryParams.size(), is(1));
    assertThat(queryParams.getAll("key"), contains("value"));
  }

  @Test
  public void onlyKeyQueryParameter() {
    io.netty.handler.codec.http.HttpRequest nettyRequest = withUri("https://test.org/path?key");
    org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest = adaptToMule(nettyRequest);

    MultiMap<String, String> queryParams = muleRequest.getQueryParams();
    assertThat(queryParams.size(), is(1));
    assertThat(queryParams.getAll("key"), contains(nullValue()));
  }

  @Test
  public void notExistentKeyQueryParameter() {
    io.netty.handler.codec.http.HttpRequest nettyRequest = withUri("https://test.org/path?key");
    org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest = adaptToMule(nettyRequest);

    MultiMap<String, String> queryParams = muleRequest.getQueryParams();
    assertThat(queryParams.size(), is(1));
    assertThat(queryParams.getAll("notExistent"), empty());
  }

  @Test
  public void multiValueForSameKeyQueryParameter() {
    io.netty.handler.codec.http.HttpRequest nettyRequest = withUri("https://test.org/path?key=value1&key=value2");
    org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest = adaptToMule(nettyRequest);

    MultiMap<String, String> queryParams = muleRequest.getQueryParams();
    assertThat(queryParams.size(), is(2));
    assertThat(queryParams.getAll("key"), contains("value1", "value2"));
  }

  @Test
  public void combinedQueryParameterTypes() {
    io.netty.handler.codec.http.HttpRequest nettyRequest =
        withUri("https://test.org/path?key1=value1_1&key1=value1_2&key2=value2&key3");
    org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest = adaptToMule(nettyRequest);

    MultiMap<String, String> queryParams = muleRequest.getQueryParams();
    assertThat(queryParams.size(), is(4));
    assertThat(queryParams.getAll("key1"), contains("value1_1", "value1_2"));
    assertThat(queryParams.getAll("key2"), contains("value2"));
    assertThat(queryParams.getAll("key3"), contains(nullValue()));
    assertThat(queryParams.getAll("key4"), empty());
  }

  @Test
  public void noQueryParameters() {
    io.netty.handler.codec.http.HttpRequest nettyRequest = withUri("https://test.org/path");
    org.mule.runtime.http.api.domain.message.request.HttpRequest muleRequest = adaptToMule(nettyRequest);

    MultiMap<String, String> queryParams = muleRequest.getQueryParams();
    assertThat(queryParams.size(), is(0));
  }

  io.netty.handler.codec.http.HttpRequest withUri(String uri) {
    return new DefaultHttpRequest(HTTP_1_1, GET, uri);
  }

  org.mule.runtime.http.api.domain.message.request.HttpRequest adaptToMule(io.netty.handler.codec.http.HttpRequest nettyRequest) {
    return new NettyHttpRequestAdapter(nettyRequest, address, testPayload);
  }
}
