/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.apache.hc.client5.http.async.methods.SimpleHttpRequest.create;
import static org.apache.hc.client5.http.impl.async.HttpAsyncClients.createHttp2Default;
import static org.apache.hc.core5.http.ContentType.TEXT_PLAIN;
import static org.apache.hc.core5.http.HttpVersion.HTTP_2;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.AllureConstants;

import java.io.IOException;

import io.qameta.allure.Feature;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;

@Feature(AllureConstants.HTTP_2)
class Http2SimpleServerTestCase extends AbstractHttpServerTestCase {

  public static final String H2_PRIOR_ECHO_ENDPOINT = "/h2-prior-echo";

  public Http2SimpleServerTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected String getServerName() {
    return "HTTP/2 Server";
  }

  @Override
  protected HttpServerConfiguration.Builder configureServer(HttpServerConfiguration.Builder builder) {
    return builder
        .setHttp1Config(new Http1ProtocolConfig(false))
        .setHttp2Config(new Http2ProtocolConfig(true));
  }

  @BeforeEach
  void setUp() throws Exception {
    setUpServer();
    server.addRequestHandler(H2_PRIOR_ECHO_ENDPOINT, (reqCtx, callback) -> {
      var echoBody = new StringHttpEntity(extractBodyAsString(reqCtx.getRequest()));
      var res = HttpResponse.builder()
          .statusCode(200)
          .entity(echoBody)
          .build();
      callback.responseReady(res, new IgnoreResponseStatusCallback());
    });
  }

  @Test
  @Disabled("TODO (W-19814153): Schedule the request handler to IO")
  void echoPostToSimpleHttp2PriorKnowledgeServer() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = create("POST", "http://localhost:" + port + H2_PRIOR_ECHO_ENDPOINT);
      request.setBody("Hello world!", TEXT_PLAIN);
      var response = h2Client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(200));
      assertThat(response.getVersion(), is(HTTP_2));
    }
  }

  @Test
  void getToSimpleHttp2PriorKnowledgeServer() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = create("GET", "http://localhost:" + port + H2_PRIOR_ECHO_ENDPOINT);
      var response = h2Client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(200));
      assertThat(response.getVersion(), is(HTTP_2));
    }
  }

  // This method consumes the full body SYNCHRONOUSLY.
  private String extractBodyAsString(HttpRequest request) {
    try {
      return new String(request.getEntity().getBytes());
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
