/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.METHOD_NOT_ALLOWED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NOT_FOUND;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.REQUEST_TOO_LONG;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.REQUEST_URI_TOO_LONG;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.SERVICE_UNAVAILABLE;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTP;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTPS;
import static org.mule.service.http.test.netty.utils.TestUtils.createServerSslContext;

import static java.util.Arrays.fill;
import static java.util.Collections.singleton;

import static org.apache.hc.core5.http.HttpStatus.SC_OK;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.TestHttp2RequestHandler;
import org.mule.service.http.test.netty.utils.client.TestSSLNettyClient;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.ExecutionException;

import io.netty.handler.ssl.SslContext;
import io.qameta.allure.Issue;
import org.apache.commons.io.IOUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

public class NettyHttpServerTestCase extends AbstractHttpTestCase {

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");
  @Rule
  public TestSSLNettyClient testClient = new TestSSLNettyClient("localhost", serverPort.getNumber(), executorRule.getExecutor());
  private HttpServer server;
  private RequestHandlerManager requestHandlerManager;
  private HttpListenerRegistry listenerRegistry;

  @Before
  public void setup()
      throws IOException, NoSuchAlgorithmException, CreateException, KeyManagementException {
    listenerRegistry = new HttpListenerRegistry();
    SslContext serverSslContext = createServerSslContext();

    int maxHeaderSectionSize = 300;

    server = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withHttpListenerRegistry(listenerRegistry)
        .withSslContext(serverSslContext)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, "test-server", true, 30000, 10000L,
                                                                           serverSslContext, maxHeaderSectionSize,
                                                                           executorRule.getExecutor()))
        .build();
    server.start();
    requestHandlerManager = server.addRequestHandler("/path", new TestHttp2RequestHandler());
    server.addRequestHandler(singleton("GET"), "/only-get", new TestHttp2RequestHandler());
  }

  @After
  public void tearDown() {
    server.stop().dispose();
  }

  @Test
  public void sendRequestToWrongPathResultsInAResponseWithStatusNotFound()
      throws ExecutionException, InterruptedException, IOException {
    HttpResponse notFoundResponse = testClient.sendGet("/not-existent");
    assertThat(notFoundResponse.getStatusCode(), is(NOT_FOUND.getStatusCode()));
  }

  @Test
  public void twoSimpleGetRequestsToTheSamePathAreRespondedCorrectly()
      throws IOException, ExecutionException, InterruptedException {
    HttpResponse response1 = testClient.sendGet("/path");
    String responseAsString1 = IOUtils.toString(response1.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response1.getStatusCode(), is(OK.getStatusCode()));
    assertThat(responseAsString1, is("Test body"));

    HttpResponse response2 = testClient.sendGet("/path");
    String responseAsString2 = IOUtils.toString(response2.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response2.getStatusCode(), is(OK.getStatusCode()));
    assertThat(responseAsString2, is("Test body"));
  }

  @Test
  public void simpleGetRequestWithInvalidHeaders()
      throws IOException, ExecutionException, InterruptedException {
    MultiMap.StringMultiMap headers = new MultiMap.StringMultiMap();
    headers.put("x-authentication-properties", "{\n\"key1\":\"value\"\n}");
    HttpResponse response1 = testClient.sendGet("/path", headers);
    String responseAsString1 = IOUtils.toString(response1.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response1.getStatusCode(), is(OK.getStatusCode()));
    assertThat(responseAsString1, is("Test body"));
  }

  @Test
  public void ifTheRequestHandlerIsStoppedThenWeExpectA503() throws ExecutionException, InterruptedException {
    requestHandlerManager.stop();

    HttpResponse response = testClient.sendGet("/path");
    assertThat(response.getStatusCode(), is(SERVICE_UNAVAILABLE.getStatusCode()));
  }

  @Test
  public void ifTheRequestHandlerIsDisposedThenWeExpectA404() throws ExecutionException, InterruptedException {
    requestHandlerManager.dispose();

    HttpResponse notFoundResponse = testClient.sendGet("/path");
    assertThat(notFoundResponse.getStatusCode(), is(NOT_FOUND.getStatusCode()));
  }

  @Test
  public void twoSimplePostRequestsToTheSamePathAreRespondedCorrectly()
      throws IOException, ExecutionException, InterruptedException {
    HttpResponse response1 = testClient.sendPost("/path", "This is a Request");
    String responseAsString1 = IOUtils.toString(response1.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response1.getStatusCode(), is(OK.getStatusCode()));
    assertThat(responseAsString1, is("Test body"));

    HttpResponse response2 = testClient.sendPost("/path", "This is a Request");
    String responseAsString2 = IOUtils.toString(response2.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response2.getStatusCode(), is(OK.getStatusCode()));
    assertThat(responseAsString2, is("Test body"));
  }

  @Test
  public void sendGetToOnlyGetEndpoint() throws ExecutionException, InterruptedException, IOException {
    HttpResponse response = testClient.sendGet("/only-get");
    String responseAsString = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    assertThat(responseAsString, is("Test body"));
  }

  @Test
  public void sendPostToOnlyGetEndpoint() throws ExecutionException, InterruptedException, IOException {
    HttpResponse response = testClient.sendPost("/only-get", "Test payload");
    assertThat(response.getStatusCode(), is(METHOD_NOT_ALLOWED.getStatusCode()));
  }

  @Test
  @Issue("W-15816690")
  public void serverWithSslContextReturnsHttpsAsProtocol()
      throws NoSuchAlgorithmException, CreateException, KeyManagementException {
    SslContext serverSslContext = createServerSslContext();
    HttpServer serverWithSslContext = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withHttpListenerRegistry(listenerRegistry)
        .withSslContext(serverSslContext)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, "test-server", true, 30000, 10000L,
                                                                           serverSslContext, executorRule.getExecutor()))
        .build();

    assertThat(serverWithSslContext.getProtocol(), is(HTTPS));
  }

  @Test
  @Issue("W-15816690")
  public void serverWithoutSslContextReturnsHttpAsProtocol() {
    HttpServer serverWithoutSslContext = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withHttpListenerRegistry(listenerRegistry)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, "test-server", true, 30000, 10000L,
                                                                           null, executorRule.getExecutor()))
        .build();

    assertThat(serverWithoutSslContext.getProtocol(), is(HTTP));
  }

  @Test
  @Issue("W-15867819")
  public void nonStartedServerDoesNotFailWithNPEToAddARequestHandler() {
    // This test was added for backwards compatibility, since the OAuth client can add a request handler before the server
    // is started, and this case was working for the Grizzly implementation.

    // Just create a server
    HttpServer notStartedServer = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withShutdownTimeout(() -> 5000L)
        .withHttpListenerRegistry(listenerRegistry)
        .build();

    // notice that we aren't starting it...

    RequestHandlerManager manager = notStartedServer.addRequestHandler("/testPath", (requestContext, responseCallback) -> {
      // do nothing
    });

    assertThat(manager, is(notNullValue()));
  }

  @Test
  @Issue("W-15631509")
  public void getRequestWithTooLargeEntityReturns413()
      throws IOException, ExecutionException, InterruptedException {
    MultiMap.StringMultiMap headers = new MultiMap.StringMultiMap();
    for (int i = 1; i <= 20; i++) {
      headers.put("testheader" + i, "testvalue" + i);
    }

    HttpResponse response = testClient.sendGet("/path", headers);
    String responseAsString = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response.getStatusCode(), is(REQUEST_TOO_LONG.getStatusCode()));
    assertThat(responseAsString, containsString("Request entity too large"));
  }

  @Test
  @Issue("W-15631509")
  public void getRequestWithLongUriReturnsDetailed414()
      throws IOException, ExecutionException, InterruptedException {
    char[] chars = new char[310];
    fill(chars, 'A');
    String longUri = "/path" + new String(chars);

    HttpResponse response = testClient.sendGet(longUri);
    String responseAsString = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(response.getStatusCode(), is(REQUEST_URI_TOO_LONG.getStatusCode()));
    assertThat(responseAsString, containsString("Request too long"));
  }

  @Test
  @Issue("W-15631497")
  public void testSSLConnection() throws Exception {
    String sslEndpoint = "/path";
    HttpResponse response = testClient.sendGet(sslEndpoint);
    String responseBody = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat("Expected response status code to be 200", response.getStatusCode(), is(SC_OK));
    assertThat("Expected response body", responseBody, is("Test body"));
  }

  @Test
  @Issue("W-15631497")
  public void testSSLRehandshake() throws Exception {
    String sslEndpoint = "/path";
    HttpResponse initialResponse = testClient.sendGet(sslEndpoint);
    assertThat(initialResponse.getStatusCode(), is(SC_OK));
    testClient.reHandshake();
    HttpResponse rehandshakeResponse = testClient.sendGet(sslEndpoint);
    assertThat(rehandshakeResponse.getStatusCode(), is(SC_OK));
  }
}
