/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.runtime.api.metadata.MediaType.TEXT;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;

import static java.lang.String.format;
import static java.lang.Thread.sleep;
import static java.util.Collections.singletonList;
import static java.util.concurrent.Executors.newCachedThreadPool;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.fail;

import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServerConfiguration;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.hc.client5.http.fluent.Request;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.SetSystemProperty;

public class SlowResponsesHandlingTestCase extends AbstractHttpServerTestCase {

  private static final long TIME_PER_BYTE = 100;
  private static final int REQUESTS = 100;
  private static final int LATCH_TIMEOUT = 5000;
  private static final String ENDPOINT = "/firstSlow";
  private final AtomicInteger requestNumber = new AtomicInteger(0);
  private final CountDownLatch start = new CountDownLatch(1);
  private final CountDownLatch latch = new CountDownLatch(REQUESTS);
  private final ExecutorService executor = newCachedThreadPool();

  public SlowResponsesHandlingTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected void setUpServer() throws Exception {
    server = service.getServerFactory().create(configureServer(new HttpServerConfiguration.Builder()
        .setHost("localhost")
        .setPort(port)
        .setName(getServerName())
        .setSchedulerSupplier(() -> getSchedulerService().ioScheduler()))
            .build());
    server.start();
  }

  @BeforeEach
  public void setUp() throws Exception {
    setUpServer();
    registerHandler("Success!");
  }

  private void registerHandler(String payload) {
    server.addRequestHandler(singletonList(GET.name()), ENDPOINT, (requestContext, responseCallback) -> {
      InputStream response;
      if (requestNumber.getAndIncrement() == 0) {
        start.countDown();
        response = new InputStream() {

          private int numreads = 0;

          @Override
          public int read() throws IOException {
            try {
              sleep(TIME_PER_BYTE);
              return numreads++ <= (LATCH_TIMEOUT / TIME_PER_BYTE + 10) ? 1 : -1;
            } catch (InterruptedException e) {
              return -1;
            }
          }
        };

      } else {
        response = new ByteArrayInputStream(payload.getBytes());
      }
      responseCallback
          .responseReady(HttpResponse.builder().entity(new InputStreamHttpEntity(response))
              .addHeader(CONTENT_TYPE, TEXT.toRfcString())
              .build(), new IgnoreResponseStatusCallback());
    }).start();
  }

  @Override
  protected String getServerName() {
    return "slow-responses";
  }

  @Test
  @SetSystemProperty(key = "mule.timeoutToUseSelectorWhileStreamingResponseMillis", value = "0")
  void onlySlowRequestFails() throws Exception {
    AtomicInteger correctCount = new AtomicInteger(0);
    performRequestInThread(null);
    start.await();
    // These should be all correct
    for (int i = 0; i < REQUESTS; i++) {
      performRequestInThread(correctCount);
    }

    if (!latch.await(LATCH_TIMEOUT, TimeUnit.MILLISECONDS)) {
      fail(format("We only got %d responses out of %d", correctCount.get(), REQUESTS));
    }
  }

  private void performRequestInThread(AtomicInteger counter) {
    executor.submit(() -> {
      try {
        Request request = Request.get(urlForPath(ENDPOINT));
        org.apache.hc.core5.http.HttpResponse response = request.execute().returnResponse();
        assertThat(response.getCode(), is(OK.getStatusCode()));
        if (counter != null) {
          counter.incrementAndGet();
          latch.countDown();
        }
      } catch (Exception e) {
        //
      }
    });
  }

}
