/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.http.api.domain.HttpProtocol.HTTP_2;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.decodeQueryString;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.extractQueryParams;

import static io.netty.buffer.ByteBufUtil.getBytes;

import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.service.http.netty.impl.server.util.DefaultServerAddress;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.netty.impl.streaming.BlockingBidirectionalStream;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.concurrent.Executor;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.codec.http2.Http2HeadersFrame;
import io.netty.handler.ssl.SslHandler;

/**
 * It receives the HTTP/2 frames and uses them to create the corresponding {@link HttpRequest} instances. Then it delegates the
 * built objects to the {@link HttpListenerRegistry} to be handled.
 */
public class NettyToMuleHttp2RequestHandlerAdapter extends SimpleChannelInboundHandler<Object> {

  public static final char PSEUDO_HEADER_PREFIX = ':';
  private final HttpListenerRegistry httpListenerRegistry;
  private final SslHandler sslHandler;
  private final Executor ioExecutor;
  private final BlockingBidirectionalStream bidiStream;

  public NettyToMuleHttp2RequestHandlerAdapter(HttpListenerRegistry httpListenerRegistry, SslHandler sslHandler,
                                               Executor ioExecutor) {
    this.httpListenerRegistry = httpListenerRegistry;
    this.sslHandler = sslHandler;
    this.ioExecutor = ioExecutor;
    this.bidiStream = new BlockingBidirectionalStream();
  }

  @Override
  protected void channelRead0(ChannelHandlerContext ctx, Object msg) throws Exception {
    if (msg instanceof Http2DataFrame dataFrame) {
      onDataFrameRead(dataFrame);
    } else if (msg instanceof Http2HeadersFrame headersFrame) {
      onHeadersFrameRead(ctx, headersFrame);
    } else {
      super.channelRead(ctx, msg);
    }
  }

  private void onHeadersFrameRead(ChannelHandlerContext ctx, Http2HeadersFrame headersFrame) throws IOException {
    if (headersFrame.isEndStream()) {
      bidiStream.getOutputStream().close();
    }

    InetSocketAddress socketAddress = (InetSocketAddress) ctx.channel().localAddress();
    DefaultServerAddress serverAddress = new DefaultServerAddress(socketAddress.getAddress(), socketAddress.getPort());
    String uriString = headersFrame.headers().path().toString();

    var entity = new InputStreamHttpEntity(bidiStream.getInputStream());

    var requestBuilder = HttpRequest.builder()
        .protocol(HTTP_2)
        .method(headersFrame.headers().method().toString())
        .uri(uriString)
        .entity(entity);

    var queryParams = decodeQueryString(extractQueryParams(uriString));
    requestBuilder.queryParams(queryParams);

    for (var entry : headersFrame.headers()) {
      var headerName = entry.getKey();
      if (headerName.charAt(0) != PSEUDO_HEADER_PREFIX) {
        var headerValue = entry.getValue();
        requestBuilder.addHeader(headerName.toString(), headerValue.toString());
      }
    }

    var request = requestBuilder.build();
    RequestHandler requestHandler = httpListenerRegistry.getRequestHandler(serverAddress, request);
    requestHandler.handleRequest(new NettyHttpRequestContext(request, ctx, sslHandler),
                                 new NettyHttp2RequestReadyCallback(ctx, headersFrame.stream(), request, ioExecutor));
  }

  private void onDataFrameRead(Http2DataFrame dataFrame) throws IOException {
    // TODO W-19810580: Avoid the copy to byte[] and try to copy directly to the output stream
    // could we even avoid the copy completely by retaining the content and using it as the data source for the InputStream?
    // would it be worth it?
    ByteBuf content = dataFrame.content();
    byte[] data = getBytes(content);
    bidiStream.getOutputStream().write(data);
    if (dataFrame.isEndStream()) {
      bidiStream.getOutputStream().close();
    }
  }
}
