/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.http.api.server.ServerNotFoundException;

import java.util.function.Supplier;

/**
 * Creates {@link HttpServer} from the {@link HttpServerConfiguration} and some additional contextual information.
 */
public interface ContextHttpServerConnectionFactory {

  /**
   * Creates a new {@link HttpServer} with the given configuration.
   *
   * @param configuration   the configuration of the new server.
   * @param context         name of the context (domain, app or policy) that creates this server.
   * @param shutdownTimeout the graceful shutdown timeout in milliseconds.
   * @return a new instance of {@link HttpServer}.
   * @throws ServerCreationException if some error happens on the creation.
   */
  HttpServer create(HttpServerConfiguration configuration, String context, Supplier<Long> shutdownTimeout)
      throws ServerCreationException;

  /**
   * Retrieves an existing {@link HttpServer} with the given name, or throws an exception if not found.
   *
   * @param name    Name of the server to lookup.
   * @param context Name of the context that created the server originally.
   *
   * @throws ServerNotFoundException if no server was found.
   */
  HttpServer lookup(String name, String context) throws ServerNotFoundException;

  /**
   * Atomically gets an existing {@link HttpServer} with the given name if it exists, or creates one with the given configuration.
   * It receives the configuration as a {@link Supplier}, so if there is no need to actually create such configuration, it doesn't
   * call the supplier.
   *
   * @param name            the name of the server to retrieve or create.
   * @param context         name of the context (domain, app or policy) that is asking for a server.
   * @param configuration   a supplier of the configuration for the new server. It's only used if the server wasn't created yet.
   * @param shutdownTimeout the graceful shutdown timeout in milliseconds.
   * @return the existing {@link HttpServer}, or a new one.
   * @throws ServerCreationException if some error happens on the creation.
   */
  HttpServer getOrCreateServer(String name, String context, Supplier<? extends HttpServerConfiguration> configuration,
                               Supplier<Long> shutdownTimeout)
      throws ServerCreationException;
}
