/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.service.http.test.netty.utils.TestUtils.preservingHeaderCase;
import static org.mule.service.http.test.netty.utils.TestUtils.sendRawRequestToServer;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.containsString;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.test.common.AbstractHttpTestCase;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.client.TestSSLNettyClient;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

public class ServerHeaderCaseSensitivityTestCase extends AbstractHttpTestCase {

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  private static final String A_RAW_REQUEST_WITH_HEADERS = """
      GET /saveRequest HTTP/1.1
      Host: localhost
      User-Agent: ObviouslyMe
      HeaderWithCase: HeaderValue

      """;

  private HttpServer server;
  private HttpListenerRegistry httpListenerRegistry;

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public TestSSLNettyClient testClient = new TestSSLNettyClient("localhost", serverPort.getNumber(), executorRule.getExecutor());

  private final BlockingQueue<HttpRequest> receivedRequests = new LinkedBlockingQueue<>();

  @Before
  public void setup()
      throws IOException, CertificateException, NoSuchAlgorithmException, CreateException, KeyManagementException {
    httpListenerRegistry = new HttpListenerRegistry();
    server = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withShutdownTimeout(() -> 5000L)
        .withHttpListenerRegistry(httpListenerRegistry)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(httpListenerRegistry, "test-server", true, 30000,
                                                                           10000L, null, executorRule.getExecutor()))
        .build();
    server.start();
    server.addRequestHandler("/saveRequest", (requestContext, responseCallback) -> {
      receivedRequests.add(requestContext.getRequest());
      responseCallback.responseReady(HttpResponse.builder().addHeader("HeaderWithCase", "HeaderValue").build(),
                                     new NoOpResponseStatusCallback());
    }).start();
  }

  @After
  public void tearDown() {
    server.stop().dispose();
  }

  @Test
  public void receivedRequestLosesHeaderCaseWhenNoPropertySet() throws IOException, InterruptedException {
    String rawResponse = sendRawRequestToServer(A_RAW_REQUEST_WITH_HEADERS, "localhost", serverPort.getNumber());

    HttpRequest requestSeenByServer = receivedRequests.take();
    assertThat(requestSeenByServer.getHeaderNames(), containsInAnyOrder("host", "user-agent", "headerwithcase"));

    assertThat(rawResponse, containsString("headerwithcase"));
  }

  @Test
  public void receivedRequestPreserveHeaderCaseWhenPropertySet() {
    preservingHeaderCase(() -> {
      String rawResponse;
      try {
        rawResponse = sendRawRequestToServer(A_RAW_REQUEST_WITH_HEADERS, "localhost", serverPort.getNumber());
        HttpRequest requestSeenByServer = receivedRequests.take();
        assertThat(requestSeenByServer.getHeaderNames(), containsInAnyOrder("Host", "User-Agent", "HeaderWithCase"));
      } catch (IOException | InterruptedException e) {
        throw new RuntimeException(e);
      }

      assertThat(rawResponse, containsString("HeaderWithCase"));
    });
  }
}
