/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.netty.impl.server.KeepAliveHandler.TIMEOUT_READING_REQUEST;

import static java.lang.Thread.sleep;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static io.netty.handler.codec.http.HttpResponseStatus.REQUEST_TIMEOUT;
import static org.apache.hc.client5.http.async.methods.SimpleHttpRequest.create;
import static org.apache.hc.client5.http.impl.async.HttpAsyncClients.createHttp2Default;
import static org.apache.hc.core5.http.ContentType.TEXT_PLAIN;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.test.common.client.sse.FillAndWaitStream;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.AllureConstants;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import io.qameta.allure.Feature;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.async.methods.SimpleResponseConsumer;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.HttpRequest;
import org.apache.hc.core5.http.nio.support.BasicRequestProducer;
import org.apache.hc.core5.http.nio.support.classic.AbstractClassicEntityProducer;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.Issue;

@Feature(AllureConstants.HTTP_2)
class Http2ServerClientDisconnectTestCase extends AbstractHttpServerTestCase {

  private static final String TEST_ENDPOINT = "/test";
  private static final int SMALL_TIMEOUT_MS = 1000;
  private static final int CONN_IDLE_TIMEOUT_MS = 1000;
  private static final int CONN_READ_TIMEOUT_MS = 300;

  private final Latch requestReceived = new Latch();
  private final Latch requestCanComplete = new Latch();

  public Http2ServerClientDisconnectTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  protected boolean getHttp1Support() {
    return false;
  }

  @Override
  protected String getServerName() {
    return "HTTP/2 Server";
  }

  @Override
  protected HttpServerConfiguration.Builder configureServer(HttpServerConfiguration.Builder builder) {
    return builder
        .setReadTimeout(CONN_READ_TIMEOUT_MS)
        .setConnectionIdleTimeout(CONN_IDLE_TIMEOUT_MS)
        .setHttp1Config(new Http1ProtocolConfig(getHttp1Support()))
        .setHttp2Config(new Http2ProtocolConfig(true));
  }

  @BeforeEach
  void setUp() throws Exception {
    setUpServer();
    server.addRequestHandler(TEST_ENDPOINT, (reqCtx, callback) -> {
      requestReceived.release();
      getSchedulerService().ioScheduler().submit(() -> {
        try {
          requestCanComplete.await(SMALL_TIMEOUT_MS, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
          throw new RuntimeException(e);
        }
        callback.responseReady(HttpResponse.builder().build(), new IgnoreResponseStatusCallback());
      });
    });
  }

  @Test
  @Issue("W-19861068")
  void whenClientDisconnectsThenServerClosesConnection() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = create("GET", urlForPath(TEST_ENDPOINT));

      // Start the request and wait for completion
      requestCanComplete.release();
      h2Client.execute(request, new IgnoreFutureCallback<>()).get();
    }
    // The client will be auto-closed at this point and the connection should be closed server side too
    assertNoActiveConnections();
  }

  @Test
  @Issue("W-19861068")
  void whenClientDisconnectsDuringRequestThenServerClosesConnection() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      // Send a request to the slow endpoint
      var request = create("GET", urlForPath(TEST_ENDPOINT));

      // Start the request but don't wait for completion
      h2Client.execute(request, new IgnoreFutureCallback<>());

      assertThat("Request was not received", requestReceived.await(SMALL_TIMEOUT_MS, TimeUnit.SECONDS), is(true));

      // Close the client connection while the request handler is still processing
      h2Client.close();

      // Allow the request to complete
      requestCanComplete.release();
    }
    assertNoActiveConnections();
  }

  @Test
  @Issue("W-19861068")
  void whenClientIsIdleAfterTimeoutThenServerClosesConnection() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = create("GET", urlForPath(TEST_ENDPOINT));

      // Start the request and wait for completion
      requestCanComplete.release();
      h2Client.execute(request, new IgnoreFutureCallback<>()).get();

      sleep(CONN_IDLE_TIMEOUT_MS + 100);
      assertNoActiveConnections();
    }
  }

  @Test
  @Issue("W-19861068")
  void whenReadTimeoutThenServerClosesConnection() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = new HttpPost(urlForPath(TEST_ENDPOINT));

      // Creates a streaming body that will pause until we give it the signal
      Latch finishRequest = new Latch();
      InputStream inputStream = new FillAndWaitStream(finishRequest);

      // Start the request and do not wait for completion
      Future<SimpleHttpResponse> respFuture = h2Client.execute(
                                                               new InputStreamRequestProducer(request, inputStream, TEXT_PLAIN,
                                                                                              getSchedulerService()
                                                                                                  .ioScheduler()),
                                                               SimpleResponseConsumer.create(),
                                                               new IgnoreFutureCallback<>());

      // Wait for the read timeout
      sleep(CONN_READ_TIMEOUT_MS + 100);

      // Wait for the timeout response
      SimpleHttpResponse resp = respFuture.get();
      assertThat(resp.getCode(), is(REQUEST_TIMEOUT.code()));
      assertThat(resp.getBodyText(), is(TIMEOUT_READING_REQUEST));

      // Unblock the producer and the request handlers for proper resource cleanup
      finishRequest.release();
      requestCanComplete.release();
    }

    assertNoActiveConnections();
  }

  private void assertNoActiveConnections() {
    AcceptedConnectionChannelInitializer clientChannelHandler = ((NettyHttpServer) server).getClientChannelHandler();
    assertThat("There are still open connections, expected none",
               clientChannelHandler.waitForConnectionsToBeClosed((long) SMALL_TIMEOUT_MS, MILLISECONDS),
               is(true));
  }

  private static class InputStreamRequestProducer extends BasicRequestProducer {

    private static final int INITIAL_BUFFER_SIZE = 1024;

    public InputStreamRequestProducer(HttpRequest request,
                                      InputStream inputStream,
                                      ContentType contentType,
                                      Executor executor) {
      super(request, new AbstractClassicEntityProducer(INITIAL_BUFFER_SIZE, contentType, executor) {

        @Override
        protected void produceData(ContentType contentType, OutputStream outputStream) throws IOException {
          inputStream.transferTo(outputStream);
        }
      });
    }
  }
}
