/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static java.lang.Integer.parseInt;
import static java.lang.System.currentTimeMillis;
import static java.time.ZoneId.of;
import static java.time.format.DateTimeFormatter.ofPattern;
import static java.util.Locale.US;

import static io.netty.handler.codec.http.HttpHeaderNames.COOKIE;
import static io.netty.handler.codec.http.HttpHeaderNames.SET_COOKIE;

import java.net.HttpCookie;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import io.netty.handler.codec.http.HttpHeaders;
import reactor.netty.http.client.HttpClientRequest;
import reactor.netty.http.client.HttpClientResponse;

/**
 * The Reactor Netty library already handles the redirection mechanism, but we need to slightly tune it to match backwards
 * compatibility requirements.
 * <p>
 * This utility class does some extra cookie parsing and validation. This class is responsible for extracting cookies from HTTP
 * headers, filtering expired cookies, and storing them into the cookies map.
 * <p>
 * This class also captures the redirect response, in order to check if the subsequent redirected request should change the method
 * as per <a href="https://tools.ietf.org/html/rfc7231#section-6.4.3">RFC 7231</a>.
 *
 * @see org.mule.service.http.netty.impl.client.RedirectMethodChangeHandler
 */
public class RedirectHelper {

  private static final Pattern MAX_AGE_PATTERN = Pattern.compile("(?i)Max-Age=([-]?\\d+)");
  private static final Pattern EXPIRES_PATTERN = Pattern.compile("(?i)Expires=([^;]+)");

  private static final DateTimeFormatter DATE_FORMATTER = ofPattern("EEE, dd MMM yyyy HH:mm:ss z", US).withZone(of("GMT"));

  private final Map<String, String> cookiesMap;
  private int responseStatusCode = -1;

  public RedirectHelper(HttpHeaders httpsHeaders) {
    this.cookiesMap = new LinkedHashMap<>();
    if (httpsHeaders != null && httpsHeaders.contains(COOKIE)) {
      List<String> cookies = httpsHeaders.getAll(COOKIE);
      cookies.stream().map(RedirectHelper::extractKeyValueAndConvertToHttpCookie).filter(Objects::nonNull)
          .forEach(parsedCookie -> cookiesMap.put(parsedCookie.getName(), parsedCookie.getValue()));
    }
  }

  /**
   * Captures cookies from redirected HTTP responses and stores them, and saves the status code.
   *
   * @param response The HTTP response containing Set-Cookie headers.
   */
  public void handleRedirectResponse(HttpClientResponse response) {
    HttpHeaders headers = response.responseHeaders();
    if (headers.contains(SET_COOKIE)) {
      filterExpiredCookies(headers);
    }
    responseStatusCode = response.status().code();
  }

  /**
   * Adds cookies from the previously sent cookie headers to the redirected HTTP request.
   *
   * @param redirectRequest The HTTP request to modify with cookies.
   */
  public void addCookiesToRedirectedRequest(HttpClientRequest redirectRequest) {
    if (cookiesMap.isEmpty()) {
      return;
    }
    StringBuilder cookieHeader = new StringBuilder();
    cookiesMap.forEach((key, value) -> {
      if (!cookieHeader.isEmpty()) {
        cookieHeader.append("; ");
      }
      cookieHeader.append(key).append("=").append(value);
    });
    redirectRequest.requestHeaders().set(COOKIE, cookieHeader.toString());
  }

  /**
   * Redirected requests should change the method to {@code GET} when status code is {@code 303}. In case of {@code 302}, the RFC
   * indicates that we MAY change the method, but we were changing it in our legacy implementation, so we will keep that behavior.
   *
   * @return {@code true} if the redirection response was a {@code 302} or {@code 303}, or {@code false} otherwise.
   */
  public Boolean shouldChangeMethod() {
    return 302 == responseStatusCode || 303 == responseStatusCode;
  }

  /**
   * @param statusCode the response status code.
   * @return whether the status code is in the range of redirection status codes (3xx).
   */
  public boolean isRedirectStatusCode(int statusCode) {
    return statusCode >= 300 && statusCode <= 399;
  }

  /**
   * Function to parse cookies which contains Expires attribute. Since {@link HttpCookie} does not support cookies which contains
   * the Expires attribute, we need to manually parse the cookie string and build a {@link HttpCookie} from it.
   */
  private static HttpCookie extractKeyValueAndConvertToHttpCookie(String cookieString) {
    String[] parts = cookieString.split(";\\s*");
    String[] nameValue = parts[0].split("=", 2);
    if (nameValue.length != 2) {
      return null;
    }
    return new HttpCookie(nameValue[0].trim(), nameValue[1].trim());
  }

  private static boolean isCookieExpired(String setCookieHeader) {
    Matcher maxAgeMatcher = MAX_AGE_PATTERN.matcher(setCookieHeader);
    Matcher expiresMatcher = EXPIRES_PATTERN.matcher(setCookieHeader);
    Long expiryTime = null;
    long currentTime = currentTimeMillis();

    if (maxAgeMatcher.find()) {
      int maxAge = parseInt(maxAgeMatcher.group(1));
      if (maxAge <= 0) {
        return true;
      }
      expiryTime = currentTime + (maxAge * 1000L);
    } else if (expiresMatcher.find()) {
      String expiresValue = expiresMatcher.group(1).trim();
      expiryTime = parseExpiresDate(expiresValue);
    }
    return expiryTime != null && currentTime > expiryTime;
  }

  private static Long parseExpiresDate(String expiresValue) {
    ZonedDateTime zdt = ZonedDateTime.parse(expiresValue, DATE_FORMATTER);
    return zdt.toInstant().toEpochMilli();
  }

  private void filterExpiredCookies(HttpHeaders headers) {
    List<String> cookieStrings = headers.getAll(SET_COOKIE);
    cookieStrings.stream().filter(cookieString -> cookieString != null && !isCookieExpired(cookieString))
        .forEach(cookieString -> {
          if (cookieString.contains("Expires=")) {
            HttpCookie httpCookie = extractKeyValueAndConvertToHttpCookie(cookieString);
            if (httpCookie != null) {
              this.cookiesMap.put(httpCookie.getName(), httpCookie.getValue());
            }
          } else {
            for (HttpCookie cookie : HttpCookie.parse(cookieString)) {
              this.cookiesMap.put(cookie.getName(), cookie.getValue());
            }
          }
        });
  }
}
