/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.decodeQueryString;
import static org.mule.runtime.http.api.utils.UriCache.getUriFromString;

import static java.lang.Long.parseLong;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;
import org.mule.runtime.http.api.domain.HttpProtocol;
import org.mule.runtime.http.api.domain.message.BaseHttpMessage;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.server.ServerAddress;

import java.net.URI;
import java.util.Collection;

import io.netty.handler.codec.http.HttpUtil;
import io.netty.handler.codec.http.HttpVersion;

public abstract class NettyHttpMessage extends BaseHttpMessage implements HttpRequest {

  protected final ServerAddress localAddress;
  protected final boolean isTransferEncodingChunked;
  protected final long contentLength;
  protected final io.netty.handler.codec.http.HttpRequest nettyRequest;
  private String baseUri;

  protected String method;
  protected MultiMap<String, String> queryParams;
  protected HttpProtocol protocol;
  protected URI uri;
  protected String path;

  public NettyHttpMessage(io.netty.handler.codec.http.HttpRequest nettyRequest,
                          ServerAddress localAddress) {
    super(null);
    this.nettyRequest = nettyRequest;
    this.localAddress = localAddress;
    this.isTransferEncodingChunked = HttpUtil.isTransferEncodingChunked(this.nettyRequest);

    long contentLengthAsLong = isTransferEncodingChunked ? -1L : 0L;
    String contentLengthAsString = this.nettyRequest.headers().getAsString(CONTENT_LENGTH);
    if (contentLengthAsString != null) {
      contentLengthAsLong = parseLong(contentLengthAsString);
    }
    this.contentLength = contentLengthAsLong;
  }

  @Override
  public MultiMap<String, String> getQueryParams() {
    if (queryParams == null) {
      parseUri();
    }
    return queryParams;
  }

  @Override
  public String getPath() {
    if (this.path == null) {
      parseUri();
    }
    return this.path;
  }

  @Override
  public String getMethod() {
    if (this.method == null) {
      this.method = nettyRequest.method().name();
    }
    return this.method;
  }

  @Override
  public HttpProtocol getProtocol() {
    if (this.protocol == null) {
      this.protocol = nettyRequest.protocolVersion() == HttpVersion.HTTP_1_0 ? HttpProtocol.HTTP_1_0 : HttpProtocol.HTTP_1_1;
    }
    return this.protocol;
  }

  @Override
  public Collection<String> getHeaderNames() {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers.keySet();
  }

  @Override
  public String getHeaderValue(String headerName) {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers.get(headerName);
  }

  @Override
  public Collection<String> getHeaderValues(String headerName) {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers.getAll(headerName);
  }

  @Override
  public MultiMap<String, String> getHeaders() {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers;
  }

  @Override
  public boolean containsHeader(String headerName) {
    return this.getHeaders().containsKey(headerName);
  }

  private void initializeHeaders() {
    this.headers = new CaseInsensitiveMultiMap(!PRESERVE_HEADER_CASE);
    this.nettyRequest.headers().forEach(entry -> this.headers.put(entry.getKey(), entry.getValue()));
    this.headers = this.headers.toImmutableMultiMap();
  }

  @Override
  public URI getUri() {
    if (this.uri == null) {
      parseUri();
    }
    return this.uri;
  }

  private void parseUri() {
    this.baseUri = getBaseProtocol() + "://" + localAddress.getAddress().getHostAddress() + ":" + localAddress.getPort();

    String fullUri = nettyRequest.uri();
    if (!nettyRequest.uri().startsWith("http")) {
      fullUri = baseUri + nettyRequest.uri();
    }

    this.uri = getUriFromString(fullUri);
    this.path = uri.getRawPath();
    this.queryParams = decodeQueryString(uri.getRawQuery());
  }

  protected abstract String getBaseProtocol();
}
