/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.Method.POST;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTP;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTPS;
import static org.mule.runtime.http.api.client.auth.HttpAuthentication.basic;
import static org.mule.service.http.test.netty.AllureConstants.HttpStory.TLS;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;

import io.qameta.allure.Description;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Story(TLS)
class HttpsClientSimpleTestCase extends AbstractHttpClientTestCase {

  private static final String PASS = "mulepassword";

  private HttpClient client;

  public HttpsClientSimpleTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void createClient() throws CreateException {
    client = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setTlsContextFactory(TlsContextFactory.builder()
            .trustStorePath("tls/trustStore")
            .trustStorePassword(PASS)
            .build())
        .setName("simple-client-tls-test")
        .build());
    client.start();
  }

  @AfterEach
  void stopClient() {
    if (client != null) {
      client.stop();
    }
  }

  @Override
  protected HttpResponse setUpHttpResponse(HttpRequest request) {
    HttpResponseBuilder response = HttpResponse.builder();
    return response.statusCode(OK.getStatusCode())
        .entity(new StringHttpEntity("OK"))
        .build();
  }

  @Test
  @Description("Send request using TLS")
  void sendRequestUsingCustomTlsProvider() throws Exception {
    var requestOptions = HttpRequestOptions.builder()
        .authentication(basic("user", "password").build())
        .build();
    var response = client.send(HttpRequest.builder()
        .method(POST)
        .uri(getUri())
        .addHeader("Transfer-Encoding", "chunked")
        .entity(new StringHttpEntity("Hola"))
        .build(), requestOptions);
    assertThat(IOUtils.toString(response.getEntity().getContent(), UTF_8), is(equalTo("OK")));
  }

  @Override
  protected HttpServerConfiguration.Builder getServerConfigurationBuilder() throws Exception {
    return super.getServerConfigurationBuilder().setTlsContextFactory(TlsContextFactory.builder()
        .keyStorePath("tls/serverKeystore")
        .keyStorePassword(PASS)
        .keyPassword(PASS)
        .build());
  }

  @Override
  protected String getUri() {
    return super.getUri().replace(HTTP.getScheme(), HTTPS.getScheme());
  }

}
