/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static io.netty.buffer.ByteBufUtil.writeUtf8;
import static io.netty.handler.codec.http.HttpResponseStatus.BAD_REQUEST;
import static io.netty.handler.codec.http2.Http2CodecUtil.HTTP_UPGRADE_PROTOCOL_NAME;
import static io.netty.util.AsciiString.contentEquals;

import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpHeaders;

import java.util.concurrent.Executor;

import io.netty.handler.codec.http.HttpServerUpgradeHandler;
import io.netty.handler.codec.http2.Http2FrameCodecBuilder;
import io.netty.handler.codec.http2.Http2MultiplexHandler;
import io.netty.handler.codec.http2.Http2ServerUpgradeCodec;

import java.util.Collection;

class UpgradeToHttp2CleartextCodecFactory implements HttpServerUpgradeHandler.UpgradeCodecFactory {

  private final HttpListenerRegistry httpListenerRegistry;
  private final Executor ioExecutor;

  public UpgradeToHttp2CleartextCodecFactory(HttpListenerRegistry httpListenerRegistry, Executor ioExecutor) {
    this.httpListenerRegistry = httpListenerRegistry;
    this.ioExecutor = ioExecutor;
  }

  @Override
  public HttpServerUpgradeHandler.UpgradeCodec newUpgradeCodec(CharSequence protocol) {
    if (contentEquals(HTTP_UPGRADE_PROTOCOL_NAME, protocol)) {
      return new UpdateCodecWithErrorResponse(new Http2ServerUpgradeCodec(Http2FrameCodecBuilder.forServer()
          .build(), new Http2MultiplexHandler(new MultiplexerChannelInitializer(httpListenerRegistry, null, ioExecutor))));
    } else {
      return null;
    }
  }

  private static final class UpdateCodecWithErrorResponse implements HttpServerUpgradeHandler.UpgradeCodec {

    private final HttpServerUpgradeHandler.UpgradeCodec delegate;

    private UpdateCodecWithErrorResponse(HttpServerUpgradeHandler.UpgradeCodec delegate) {
      this.delegate = delegate;
    }

    @Override
    public Collection<CharSequence> requiredUpgradeHeaders() {
      return delegate.requiredUpgradeHeaders();
    }

    @Override
    public boolean prepareUpgradeResponse(ChannelHandlerContext ctx, FullHttpRequest upgradeRequest, HttpHeaders upgradeHeaders) {
      if (delegate.prepareUpgradeResponse(ctx, upgradeRequest, upgradeHeaders)) {
        return true;
      }
      ByteBuf responseMsg = Unpooled.buffer();
      writeUtf8(responseMsg, "HTTP Upgrade request to HTTP2 failed");
      ctx.writeAndFlush(new DefaultFullHttpResponse(upgradeRequest.protocolVersion(), BAD_REQUEST, responseMsg));
      ctx.fireExceptionCaught(new IllegalArgumentException("HTTP Upgrade Request was rejected"));
      return false;

    }

    @Override
    public void upgradeTo(ChannelHandlerContext ctx, FullHttpRequest upgradeRequest) {
      delegate.upgradeTo(ctx, upgradeRequest);
    }
  }
}
