/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server.util;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;

import static java.lang.Integer.getInteger;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.PathAndMethodRequestMatcher;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.runtime.http.api.server.ServerAddress;
import org.mule.runtime.http.api.utils.RequestMatcherRegistry;
import org.mule.service.http.netty.impl.server.RequestEntityTooLargeHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import io.netty.handler.codec.TooLongFrameException;
import io.netty.handler.codec.http.TooLongHttpLineException;
import org.slf4j.Logger;

/**
 * Registry of servers and its handlers, which allows searching for handlers and introducing new ones (while allowing them to be
 * managed).
 */
public class HttpListenerRegistry implements RequestHandlerProvider {

  private static final Logger LOGGER = getLogger(HttpListenerRegistry.class);
  private static final int MAX_NUM_HEADERS_DEFAULT = 100;
  private static final String MAX_SERVER_REQUEST_HEADERS_KEY = SYSTEM_PROPERTY_PREFIX + "http.MAX_SERVER_REQUEST_HEADERS";
  private static int MAX_SERVER_REQUEST_HEADERS =
      getInteger(MAX_SERVER_REQUEST_HEADERS_KEY, MAX_NUM_HEADERS_DEFAULT);

  private final ServerAddressMap<HttpServer> serverAddressToServerMap = new ServerAddressMap<>();
  private final Map<HttpServer, RequestMatcherRegistry<RequestHandler>> requestHandlerPerServerAddress = new HashMap<>();
  private final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
  private final Lock readLock = readWriteLock.readLock();
  private final Lock writeLock = readWriteLock.writeLock();

  /**
   * Introduces a new {@link RequestHandler} for requests matching a given {@link PathAndMethodRequestMatcher} in the provided
   * {@link HttpServer}.
   *
   * @param server         where the handler should be added
   * @param requestHandler the handler to add
   * @param requestMatcher the matcher to be applied for the handler
   * @return a {@link RequestHandlerManager} for the added handler that allows enabling, disabling and disposing it
   */
  public RequestHandlerManager addRequestHandler(final HttpServer server,
                                                 final RequestHandler requestHandler,
                                                 final PathAndMethodRequestMatcher requestMatcher) {
    writeLock.lock();
    try {
      RequestMatcherRegistry<RequestHandler> serverAddressRequestHandlerRegistry =
          this.requestHandlerPerServerAddress.get(server);
      if (serverAddressRequestHandlerRegistry == null) {
        serverAddressRequestHandlerRegistry = new DefaultRequestMatcherRegistryBuilder<RequestHandler>()
            .onMethodMismatch(NoMethodRequestHandler::getInstance)
            .onNotFound(NoListenerRequestHandler::getInstance)
            .onInvalidRequest(BadRequestHandler::getInstance)
            .onDisabled(ServiceTemporarilyUnavailableListenerRequestHandler::getInstance)
            .build();
        requestHandlerPerServerAddress.put(server, serverAddressRequestHandlerRegistry);
        serverAddressToServerMap.put(server.getServerAddress(), server);
      }
      return new DefaultRequestHandlerManager(serverAddressRequestHandlerRegistry.add(requestMatcher, requestHandler));
    } finally {
      writeLock.unlock();
    }
  }

  /**
   * Removes all handlers for a given {@link HttpServer}.
   *
   * @param server whose handlers will be removed
   */
  public void removeHandlersFor(HttpServer server) {
    writeLock.lock();
    try {
      requestHandlerPerServerAddress.remove(server);
      serverAddressToServerMap.remove(server.getServerAddress());
    } finally {
      writeLock.unlock();
    }
  }

  @Override
  public boolean hasHandlerFor(ServerAddress serverAddress) {
    readLock.lock();
    try {
      return serverAddressToServerMap.get(serverAddress) != null;
    } finally {
      readLock.unlock();
    }
  }

  @Override
  public RequestHandler getRequestHandler(ServerAddress serverAddress, final HttpRequest request) {
    LOGGER.debug("Looking RequestHandler for request: {}", request.getPath());
    if (!checkMaxRequestHeadersLimit(request.getHeaders())) {
      return RequestEntityTooLargeHandler.getInstance();
    }

    readLock.lock();
    try {
      final HttpServer server = serverAddressToServerMap.get(serverAddress);
      if (server != null && !server.isStopped()) {
        var serverAddressRequestHandlerRegistry = requestHandlerPerServerAddress.get(server);
        if (serverAddressRequestHandlerRegistry != null) {
          return serverAddressRequestHandlerRegistry.find(request);
        }
      }
      LOGGER.debug("No RequestHandler found for request: {}", request.getPath());
      return NoListenerRequestHandler.getInstance();
    } finally {
      readLock.unlock();
    }
  }

  private static boolean checkMaxRequestHeadersLimit(Map<String, String> headers) {
    if (headers.size() > MAX_SERVER_REQUEST_HEADERS) {
      LOGGER.warn("Exceeded max server request headers limit: {}. Current header count (including default headers): {}",
                  MAX_SERVER_REQUEST_HEADERS,
                  headers.size());
      return false;
    }
    return true;
  }

  public static void refreshMaxServerRequestHeaders() {
    MAX_SERVER_REQUEST_HEADERS = getInteger(MAX_SERVER_REQUEST_HEADERS_KEY, MAX_NUM_HEADERS_DEFAULT);;
  }

  public static int getMaxServerRequestHeaders() {
    return MAX_SERVER_REQUEST_HEADERS;
  }

  @Override
  public RequestHandler getErrorHandler(Throwable errorCause) {
    if (errorCause instanceof TooLongHttpLineException) {
      return HeaderTooLongRequestHandler.getInstance();
    }
    if (errorCause instanceof TooLongFrameException) {
      return RequestEntityTooLargeHandler.getInstance();
    }
    return null;
  }

  public RequestHandler getBadRequestHandler() {
    return BadRequestHandler.getInstance();
  }
}
