/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.proxy;

import org.mule.runtime.http.api.client.proxy.AuthHeaderFactory;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.auth.BasicAuthHeaderFactory;
import org.mule.service.http.netty.impl.client.auth.NettyAuthHeaderFactory;
import org.mule.service.http.netty.impl.client.auth.ntlm.NtlmMessageFactory;

/**
 * Utility wrapper of the {@link NettyAuthHeaderFactory} and {@link AuthHeaderFactory} corresponding to a given
 * {@link ProxyConfig}. It supports custom, basic and NTLM authentication schemes, and can also be used as a "null object" (it
 * would be an already finished authenticator that returns "none" as auth scheme).
 */
public class ProxyAuthenticator {

  private static final String AUTH_BASIC = "basic";
  private static final String AUTH_NONE = "none";
  private static final String AUTH_NTLM = "ntlm";
  private static final String AUTH_CUSTOM = "custom";

  private final String authScheme;
  private final NettyAuthHeaderFactory authHeaderFactory;
  private final AuthHeaderFactory customAuthHeaderFactory;

  public ProxyAuthenticator(ProxyConfig proxyConfig) {
    // when proxy config has non-null custom header factory
    if (proxyConfig.getAuthHeaderFactory() != null) {
      this.authScheme = AUTH_CUSTOM;
      this.authHeaderFactory = null;
      this.customAuthHeaderFactory = proxyConfig.getAuthHeaderFactory();
      return;
    }
    this.customAuthHeaderFactory = null;

    if (proxyConfig.getUsername() == null) {
      this.authScheme = AUTH_NONE;
      this.authHeaderFactory = null;
      return;
    }

    if (proxyConfig instanceof ProxyConfig.NtlmProxyConfig ntlmProxyConfig) {
      this.authScheme = AUTH_NTLM;
      this.authHeaderFactory = new NtlmMessageFactory(ntlmProxyConfig.getNtlmDomain(), null, ntlmProxyConfig.getUsername(),
                                                      ntlmProxyConfig.getPassword());
      return;
    }

    this.authScheme = AUTH_BASIC;
    this.authHeaderFactory = new BasicAuthHeaderFactory(true, proxyConfig.getUsername(), proxyConfig.getPassword());
  }

  /**
   * @return the authentication scheme. It can be {@code "basic"}, {@code "ntlm"}, {@code "custom"}, or {@code "none"}.
   */
  public String getAuthScheme() {
    return authScheme;
  }

  /**
   * For the special case of a "none" authenticator, this method always returns {@code true}. For custom authentication, delegates
   * to the custom factory's hasFinished() method, or returns {@code true} if no factory is set. For the other schemes (basic,
   * NTLM), see {@link NettyAuthHeaderFactory#hasFinished()}.
   *
   * @return {@code true} if the authentication has finished.
   * @see NettyAuthHeaderFactory#hasFinished().
   */
  public boolean hasFinished() {
    // prioritize custom case first
    if (AUTH_CUSTOM.equals(authScheme)) {
      if (customAuthHeaderFactory == null) {
        return true;
      }
      return customAuthHeaderFactory.hasFinished();
    }

    if (authHeaderFactory == null) {
      return true;
    }
    return authHeaderFactory.hasFinished();
  }

  /**
   * For the special case of a "none" authenticator, this method always returns {@code null}. For the other schemes, see
   * {@link NettyAuthHeaderFactory#getNextHeader(String)}}.
   *
   * @param receivedAuthenticateHeader the "Proxy-Authenticate" header sent by the server.
   * @return the "Proxy-Authorization" header that should be sent by the client.
   * @throws Exception if an error occurs while calculating the header value.
   */
  public String getNextHeader(String receivedAuthenticateHeader) throws Exception {
    if (authHeaderFactory == null) {
      return null;
    }
    return authHeaderFactory.getNextHeader(receivedAuthenticateHeader);
  }

  /**
   * Generates the next "Proxy-Authorization" header for custom authentication.
   *
   * @param response the HTTP response
   * @return the "Proxy-Authorization" header that should be sent by the client
   * @throws Exception if an error occurs while calculating the header value
   */
  public String getNextHeader(HttpResponse response) throws Exception {
    if (customAuthHeaderFactory == null) {
      return null;
    }
    return customAuthHeaderFactory.generateHeader(response).get();
  }
}
