/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.METHOD_NOT_ALLOWED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NOT_FOUND;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.REQUEST_TOO_LONG;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.REQUEST_URI_TOO_LONG;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.SERVICE_UNAVAILABLE;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTP;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTPS;
import static org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer.MAXIMUM_HEADER_SECTION_SIZE_PROPERTY_KEY;
import static org.mule.service.http.test.netty.impl.server.ServerGracefulShutdownTestCase.executorRule;
import static org.mule.service.http.test.netty.utils.TestUtils.createServerSslContext;

import static java.util.Arrays.fill;
import static java.util.Collections.singleton;

import static org.apache.hc.client5.http.async.methods.SimpleHttpRequest.create;
import static org.apache.hc.client5.http.impl.async.HttpAsyncClients.createDefault;
import static org.apache.hc.core5.http.ContentType.TEXT_PLAIN;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;

import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.utils.DummyRequestHandler;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Issue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.SetSystemProperty;

class NettyHttpServerTestCase extends AbstractHttpServerTestCase {

  private RequestHandlerManager requestHandlerManager;

  public NettyHttpServerTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected String getServerName() {
    return "test-server";
  }

  @BeforeEach
  void setup() throws Exception {
    setUpServer();
    requestHandlerManager = server.addRequestHandler("/path", new DummyRequestHandler());
    requestHandlerManager.start();
    server.addRequestHandler(singleton("GET"), "/only-get", new DummyRequestHandler()).start();
  }

  @Test
  void sendRequestToWrongPathResultsInAResponseWithStatusNotFound()
      throws ExecutionException, InterruptedException, IOException {
    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath("/not-existent"));
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(NOT_FOUND.getStatusCode()));
    }
  }

  @Test
  void twoSimpleGetRequestsToTheSamePathAreRespondedCorrectly() throws Exception {
    try (var client = createDefault()) {
      client.start();

      var request1 = create("GET", urlForPath("/path"));
      var response1 = client.execute(request1, new IgnoreFutureCallback<>()).get();
      assertThat(response1.getCode(), is(OK.getStatusCode()));
      assertThat(response1.getBodyText(), is("Test body"));

      var request2 = create("GET", urlForPath("/path"));
      var response2 = client.execute(request2, new IgnoreFutureCallback<>()).get();
      assertThat(response2.getCode(), is(OK.getStatusCode()));
      assertThat(response2.getBodyText(), is("Test body"));
    }
  }

  @Test
  void simpleGetRequestWithInvalidHeaders() throws Exception {
    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath("/path"));
      request.addHeader("x-authentication-properties", "{\n\"key1\":\"value\"\n}");
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(OK.getStatusCode()));
      assertThat(response.getBodyText(), is("Test body"));
    }
  }

  @Test
  void ifTheRequestHandlerIsStoppedThenWeExpectA503() throws Exception {
    requestHandlerManager.stop();

    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath("/path"));
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(SERVICE_UNAVAILABLE.getStatusCode()));
    }
  }

  @Test
  void ifTheRequestHandlerIsDisposedThenWeExpectA404() throws Exception {
    requestHandlerManager.dispose();

    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath("/path"));
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(NOT_FOUND.getStatusCode()));
    }
  }

  @Test
  void twoSimplePostRequestsToTheSamePathAreRespondedCorrectly() throws Exception {
    try (var client = createDefault()) {
      client.start();

      var request1 = create("POST", urlForPath("/path"));
      request1.setBody("This is a Request", TEXT_PLAIN);
      var response1 = client.execute(request1, new IgnoreFutureCallback<>()).get();
      assertThat(response1.getCode(), is(OK.getStatusCode()));
      assertThat(response1.getBodyText(), is("Test body"));

      var request2 = create("POST", urlForPath("/path"));
      request2.setBody("This is a Request", TEXT_PLAIN);
      var response2 = client.execute(request2, new IgnoreFutureCallback<>()).get();
      assertThat(response2.getCode(), is(OK.getStatusCode()));
      assertThat(response2.getBodyText(), is("Test body"));
    }
  }

  @Test
  void sendGetToOnlyGetEndpoint() throws Exception {
    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath("/only-get"));
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(OK.getStatusCode()));
      assertThat(response.getBodyText(), is("Test body"));
    }
  }

  @Test
  void sendPostToOnlyGetEndpoint() throws Exception {
    try (var client = createDefault()) {
      client.start();

      var request = create("POST", urlForPath("/only-get"));
      request.setBody("Test payload", TEXT_PLAIN);
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(METHOD_NOT_ALLOWED.getStatusCode()));
    }
  }

  @Test
  @Issue("W-15816690")
  void serverWithSslContextReturnsHttpsAsProtocol() throws Exception {
    var listenerRegistry = new HttpListenerRegistry();
    var serverSslContext = createServerSslContext();
    var serverWithSslContext = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(port))
        .withHttpListenerRegistry(listenerRegistry)
        .withSslContext(serverSslContext)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, "test-server", "localhost",
                                                                           port, true, 30000, 10000L,
                                                                           serverSslContext, executorRule.getExecutor()))
        .build();

    assertThat(serverWithSslContext.getProtocol(), is(HTTPS));
  }

  @Test
  @Issue("W-15816690")
  void serverWithoutSslContextReturnsHttpAsProtocol() {
    var listenerRegistry = new HttpListenerRegistry();
    var serverWithoutSslContext = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(port))
        .withHttpListenerRegistry(listenerRegistry)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, "test-server", "localhost",
                                                                           port, true, 30000, 10000L,
                                                                           null, executorRule.getExecutor()))
        .build();

    assertThat(serverWithoutSslContext.getProtocol(), is(HTTP));
  }

  @Test
  @Issue("W-15867819")
  void nonStartedServerDoesNotFailWithNPEToAddARequestHandler() {
    // This test was added for backwards compatibility, since the OAuth client can add a request handler before the server
    // is started, and this case was working for the Grizzly implementation.

    var listenerRegistry = new HttpListenerRegistry();

    // Just create a server
    var notStartedServer = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(port))
        .withShutdownTimeout(() -> 5000L)
        .withHttpListenerRegistry(listenerRegistry)
        .build();

    // notice that we aren't starting it...

    RequestHandlerManager manager = notStartedServer.addRequestHandler("/testPath", (requestContext, responseCallback) -> {
      // do nothing
    });

    assertThat(manager, is(notNullValue()));
  }

  @Test
  @Issue("W-15631509")
  @SetSystemProperty(key = MAXIMUM_HEADER_SECTION_SIZE_PROPERTY_KEY, value = "300")
  void getRequestWithTooLargeEntityReturns413() throws Exception {
    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath("/path"));
      for (int i = 1; i <= 20; i++) {
        request.addHeader("testheader" + i, "testvalue" + i);
      }
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(REQUEST_TOO_LONG.getStatusCode()));
      assertThat(response.getBodyText(), containsString("Request entity too large"));
    }
  }

  @Test
  @Issue("W-15631509")
  @SetSystemProperty(key = MAXIMUM_HEADER_SECTION_SIZE_PROPERTY_KEY, value = "300")
  void getRequestWithLongUriReturnsDetailed414() throws Exception {
    char[] chars = new char[310];
    fill(chars, 'A');
    String longUri = "/path" + new String(chars);

    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath(longUri));
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(REQUEST_URI_TOO_LONG.getStatusCode()));
      assertThat(response.getBodyText(), containsString("Request too long"));
    }
  }

  @Test
  @Issue("W-15631497")
  void testSSLConnection() throws Exception {
    String sslEndpoint = "/path";

    try (var client = createDefault()) {
      client.start();

      var request = create("GET", urlForPath(sslEndpoint));
      var response = client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat("Expected response status code to be 200", response.getCode(), is(OK.getStatusCode()));
      assertThat("Expected response body", response.getBodyText(), is("Test body"));
    }
  }
}
