/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static java.lang.Thread.sleep;
import static java.util.concurrent.Executors.newSingleThreadScheduledExecutor;

import static org.apache.hc.client5.http.async.methods.SimpleHttpRequest.create;
import static org.apache.hc.client5.http.impl.async.HttpAsyncClients.createDefault;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServerConfiguration;

import java.util.concurrent.ScheduledExecutorService;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.Issue;

@Issue("W-19810563")
public class HttpServerConnectionTimeoutTestCase extends AbstractHttpServerTestCase {

  private static final int CONNECTION_IDLE_TIMEOUT = 1000;
  private static final String REQUEST_PATH = "/slower-than-idle-connection-timeout";

  private ScheduledExecutorService executorService;

  public HttpServerConnectionTimeoutTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  public void setup() throws Exception {
    executorService = newSingleThreadScheduledExecutor();

    setUpServer();
    server.addRequestHandler(REQUEST_PATH, (reqCtx, respCb) -> {
      executorService.submit(() -> {
        try {
          long longerThanConnectionTimeout = CONNECTION_IDLE_TIMEOUT + 100;
          sleep(longerThanConnectionTimeout);
          var res = HttpResponse.builder().statusCode(200).build();
          respCb.responseReady(res, new IgnoreResponseStatusCallback());
        } catch (InterruptedException e) {
          Thread.currentThread().interrupt();
          var res = HttpResponse.builder().statusCode(500).build();
          respCb.responseReady(res, new IgnoreResponseStatusCallback());
        }
      });
    });
  }

  @AfterEach
  void shutdownExecutorService() {
    executorService.shutdownNow();
  }

  @Override
  protected String getServerName() {
    return "small-idle-timeout";
  }

  @Override
  protected HttpServerConfiguration.Builder configureServer(HttpServerConfiguration.Builder builder) {
    return builder.setConnectionIdleTimeout(CONNECTION_IDLE_TIMEOUT);
  }

  @Test
  public void connectionTimeoutShouldNotInterruptInflightRequestProcessing() throws Exception {
    try (var httpClient = createDefault()) {
      httpClient.start();

      var request = create("GET", urlForPath(REQUEST_PATH));
      var response = httpClient.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(200));
    }
  }
}
