/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.utils;

import static java.util.Base64.getDecoder;
import static java.util.Base64.getEncoder;

import static org.eclipse.jetty.http.HttpHeader.AUTHORIZATION;
import static org.eclipse.jetty.http.HttpHeader.WWW_AUTHENTICATE;

import java.io.IOException;

import org.eclipse.jetty.security.ServerAuthException;
import org.eclipse.jetty.security.authentication.LoginAuthenticator;
import org.eclipse.jetty.server.Authentication;

import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jcifs.ntlmssp.Type1Message;
import jcifs.ntlmssp.Type2Message;

public class NtlmLoginAuthenticator extends LoginAuthenticator {

  public static final String NTLM_HEADER_PREFIX = "NTLM ";

  @Override
  public String getAuthMethod() {
    return "NTLM";
  }

  @Override
  public Authentication validateRequest(ServletRequest servletRequest, ServletResponse servletResponse, boolean b)
      throws ServerAuthException {
    try {
      HttpServletRequest request = (HttpServletRequest) servletRequest;
      HttpServletResponse response = (HttpServletResponse) servletResponse;
      String authHeader = request.getHeader(AUTHORIZATION.asString());

      if (authHeader == null) {
        response.setHeader(WWW_AUTHENTICATE.asString(), getAuthMethod());
        return Authentication.UNAUTHENTICATED;
      }

      if (isNtlmTypeN(authHeader, 1)) {
        response.setHeader(WWW_AUTHENTICATE.asString(), createType2Header(authHeader));
        return Authentication.UNAUTHENTICATED;
      }

      if (isNtlmTypeN(authHeader, 3)) {
        return Authentication.SEND_SUCCESS;
      }

      return Authentication.UNAUTHENTICATED;
    } catch (Exception e) {
      throw new ServerAuthException(e);
    }

  }

  private static boolean isNtlmTypeN(String header, int n) {
    if (!header.startsWith(NTLM_HEADER_PREFIX)) {
      return false;
    }

    String base64 = header.substring(5);

    byte[] asByteArray = getDecoder().decode(base64);
    byte type = asByteArray[8];
    return type == n;
  }

  String createType2Header(String type1Header) throws IOException {
    String base64 = type1Header.substring(5);
    byte[] type1AsByteArray = getDecoder().decode(base64);
    Type1Message type1Message = new Type1Message(type1AsByteArray);
    Type2Message type2Message = new Type2Message(type1Message);

    var encoded = getEncoder().encodeToString(type2Message.toByteArray());
    return NTLM_HEADER_PREFIX + encoded;
  }

  @Override
  public boolean secureResponse(ServletRequest servletRequest, ServletResponse servletResponse, boolean b,
                                Authentication.User user)
      throws ServerAuthException {
    return true;
  }
}
