/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static org.mule.runtime.api.metadata.MediaType.MULTIPART_MIXED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NOT_MODIFIED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NO_CONTENT;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.RESET_CONTENT;

import static java.lang.Long.parseLong;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.service.http.netty.impl.message.content.StreamedMultipartHttpEntity;

import java.io.InputStream;

import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpResponse;
import reactor.netty.http.client.HttpClientResponse;

public class HttpResponseCreatorUtils {

  public HttpEntity createEntity(InputStream stream, String contentType, String contentLength, int statusCode) {
    long contentLengthAsLong = parseContentLength(contentLength);

    if (isMultipartMixed(contentType)) {
      return createMultipartEntity(stream, contentLengthAsLong, contentType);
    }

    if (contentLengthAsLong > 0) {
      return new InputStreamHttpEntity(stream, contentLengthAsLong);
    } else if (contentLengthAsLong == 0 || isEmptyResponse(statusCode)) {
      return new EmptyHttpEntity();
    } else {
      return new InputStreamHttpEntity(stream);
    }
  }

  private long parseContentLength(String contentLength) {
    return contentLength != null ? parseLong(contentLength) : -1L;
  }

  private boolean isMultipartMixed(String contentType) {
    return contentType != null && contentType.startsWith(MULTIPART_MIXED.getPrimaryType());
  }

  public String extractHeader(HttpResponse response, String headerName) {
    HttpHeaders headers = response.headers();
    return headers.contains(headerName) ? headers.get(headerName) : null;
  }

  public String extractHeader(HttpClientResponse response, String headerName) {
    HttpHeaders headers = response.responseHeaders();
    return headers.get(headerName);
  }

  private HttpEntity createMultipartEntity(InputStream stream, long contentLengthAsLong, String contentType) {
    return contentLengthAsLong >= 0 ? new StreamedMultipartHttpEntity(stream, contentType, contentLengthAsLong)
        : new StreamedMultipartHttpEntity(stream, contentType);
  }

  private boolean isEmptyResponse(int statusCode) {
    return statusCode == NO_CONTENT.getStatusCode() || statusCode == NOT_MODIFIED.getStatusCode()
        || statusCode == RESET_CONTENT.getStatusCode();
  }

  public HttpResponseBuilder buildResponseHeaders(HttpResponse response) {
    HttpResponseBuilder responseBuilder =
        org.mule.runtime.http.api.domain.message.response.HttpResponse.builder().statusCode(response.status().code())
            .reasonPhrase(response.status().reasonPhrase());

    HttpHeaders headers = response.headers();
    if (!headers.isEmpty()) {
      headers.names().forEach(headerName -> responseBuilder.addHeaders(headerName, headers.getAll(headerName)));
    }
    return responseBuilder;
  }
}
