/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.benchmark;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import static org.openjdk.jmh.annotations.Mode.Throughput;

import io.netty.util.ReferenceCounted;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;

@BenchmarkMode(Throughput)
@OutputTimeUnit(MILLISECONDS)
@Fork(value = 1, jvmArgsAppend = {
// "-XX:+FlightRecorder",
// "-XX:StartFlightRecording=filename=cpu-profile.jfr,dumponexit=true,settings=<placeholder for your custom jfc>",
})
public class ServerPipelineBenchmark {

  @Benchmark
  @Threads(1)
  @Warmup(iterations = 1, time = 3, timeUnit = SECONDS)
  @Measurement(iterations = 2, time = 60, timeUnit = SECONDS)
  public void readRequests(ServerPipelineBenchmarkState state) {
    state.requestAsByteBuf.resetReaderIndex();
    state.requestAsByteBuf.retain();

    try {
      // The inbound pipeline will parse the raw request and call the listener, but at the end of the pipeline we won't see any
      // message
      state.channel.writeInbound(state.requestAsByteBuf);

      // The response message that should be written to the socket will be an outbound message, and we will see that here.
      var responseAsByteBuf = state.channel.readOutbound();
      if (responseAsByteBuf == null) {
        // Just checking for nullity here to avoid the noise in the benchmark...
        throw new IllegalStateException("No response read");
      } else {
        if (responseAsByteBuf instanceof ReferenceCounted referenceCounted) {
          referenceCounted.release();
        }
      }
    } catch (Exception e) {
      throw new RuntimeException("Request execution failed", e);
    }
  }
}
