/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.service.http.test.netty.AllureConstants.HttpStory.TIMEOUTS;
import static org.mule.service.http.test.netty.utils.TestUtils.measuringNanoseconds;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.containsStringIgnoringCase;
import static org.hamcrest.Matchers.lessThan;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;
import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.tcp.TcpClientSocketProperties;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.tck.junit5.DynamicPort;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;

import io.qameta.allure.Story;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.condition.EnabledOnOs;
import org.junit.jupiter.api.condition.OS;

@Story(TIMEOUTS)
public class ClientTcpConnectionTimeoutTestCase extends AbstractHttpServiceTestCase {

  private static final Integer CLIENT_CONNECTION_TIMEOUT_MILLIS = 1;

  @DynamicPort(systemProperty = "tcpServerPort")
  Integer tcpServerPort;

  private HttpClient client;

  public ClientTcpConnectionTimeoutTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setup() {
    var withConnectionTimeout = TcpClientSocketProperties.builder()
        .connectionTimeout(CLIENT_CONNECTION_TIMEOUT_MILLIS)
        .build();
    var clientConfig = new HttpClientConfiguration.Builder()
        .setName("client-with-connection-timeout")
        .setClientSocketProperties(withConnectionTimeout)
        .build();
    client = service.getClientFactory().create(clientConfig);
    client.start();
  }

  @AfterEach
  void tearDown() {
    if (client != null) {
      client.stop();
    }
  }

  // Backlog configuration is just a hint for Linux kernel, so I can't be sure that a connection is
  // in the backlog queue.
  // If you can, please adapt the test.
  @EnabledOnOs(OS.MAC)
  @Test
  void clientTcpConnectionTimeout() throws Exception {
    // When using Java's ServerSocket, we have two different listen backlog queues:
    // 1 - The tcp "syn" queue is the operative system one...
    // 2 - Java accepts the connections and move them to an auxiliary queue, the "accept" queue, and the accept() method will just
    // get a connection from THAT queue.

    // The serverSocket will be bound and listening within the try, but we won't call accept()
    try (var serverSocket = new ServerSocket(tcpServerPort, 1)) {
      // Sending a client socket to fill the "accept" queue (the "syn" queue will remain empty).
      try (var auxClientSocket = new Socket("localhost", tcpServerPort)) {
        // Then, our client connection will go to the syn queue of the server, and nobody will accept it
        var request = HttpRequest.builder()
            .uri("http://localhost:" + tcpServerPort)
            .build();

        long elapsedNanos = measuringNanoseconds(() -> {
          var error = assertThrows(IOException.class, () -> client.send(request));
          assertThat(error, hasMessage(anyOf(
                                             containsStringIgnoringCase("timeout"),
                                             containsStringIgnoringCase("timed out"))));
        });
        long expectedTimeoutNanos = MILLISECONDS.toNanos(CLIENT_CONNECTION_TIMEOUT_MILLIS);
        long toleranceNanos = SECONDS.toNanos(1); // default connection timeout is 30 seconds...
        assertThat(elapsedNanos, lessThan(expectedTimeoutNanos + toleranceNanos));
      }
    }
  }
}
