/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static io.netty.buffer.Unpooled.EMPTY_BUFFER;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http2.DefaultHttp2DataFrame;
import io.netty.handler.codec.http2.DefaultHttp2Headers;
import io.netty.handler.codec.http2.DefaultHttp2HeadersFrame;
import io.netty.handler.codec.http2.Http2FrameStream;
import io.netty.handler.codec.http2.Http2Headers;

/**
 * Implementation of {@link HttpWriter} using HTTP/2.
 */
public final class Http2Writer implements HttpWriter {

  private final ChannelHandlerContext ctx;
  private final Http2FrameStream frameStream;

  private HttpResponse response;

  public Http2Writer(ChannelHandlerContext ctx, Http2FrameStream frameStream) {
    this.ctx = ctx;
    this.frameStream = frameStream;
  }

  @Override
  public void writeResponseHeader(HttpRequest request, HttpResponse response, ChannelPromise promise) {
    this.response = response;
    Http2Headers headers = extractHeaders(response);
    ctx.writeAndFlush(new DefaultHttp2HeadersFrame(headers, false).stream(frameStream), promise);
  }

  @Override
  public void writeContent(ByteBuf content, boolean isLast, ChannelPromise promise) {
    ctx.writeAndFlush(new DefaultHttp2DataFrame(content, isLast).stream(frameStream), promise);
  }

  private static Http2Headers extractHeaders(HttpResponse response) {
    // Create HTTP/2 headers and copy all headers from the Mule response
    Http2Headers headers = new DefaultHttp2Headers(false).status(String.valueOf(response.getStatusCode()));

    // Copy all headers from the Mule response to HTTP/2 headers
    MultiMap<String, String> responseHeaders = response.getHeaders();
    for (String key : responseHeaders.keySet()) {
      for (String value : responseHeaders.getAll(key)) {
        headers.add(key, value);
      }
    }
    return headers;
  }

  @Override
  public void writeTrailers(MultiMap<String, String> trailers, ChannelPromise promise) {
    // We have no trailers to send, just send the last content frame
    if (trailers == null || trailers.isEmpty()) {
      ctx.writeAndFlush(new DefaultHttp2DataFrame(EMPTY_BUFFER, true).stream(frameStream), promise);
      return;
    }

    // Build the trailers frame (it's a headers frame without the pseudo-headers)
    Http2Headers trailersToNetty = new DefaultHttp2Headers(false);
    for (String key : trailers.keySet()) {
      for (String value : trailers.getAll(key)) {
        trailersToNetty.add(key, value);
      }
    }

    // And write the trailers frame with endStream=true
    ctx.writeAndFlush(new DefaultHttp2HeadersFrame(trailersToNetty, true).stream(frameStream), promise);
  }
}
