/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message.content;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;

import static org.apache.commons.io.IOUtils.toByteArray;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.entity.FeedableHttpEntity;
import org.mule.service.http.netty.impl.streaming.BlockingBidirectionalStream;
import org.mule.service.http.netty.impl.streaming.CancelableOutputStream;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;

import org.slf4j.Logger;

public class NettyFeedableHttpEntity implements FeedableHttpEntity {

  private static final Logger LOGGER = getLogger(NettyFeedableHttpEntity.class);
  private static final long UNKNOWN_CONTENT_LENGTH = -1L;

  private final InputStream asInputStream;
  private final CancelableOutputStream dataSink;
  private final long contentLength;
  private final CompletableFuture<MultiMap<String, String>> futureTrailers;

  public NettyFeedableHttpEntity() {
    this(UNKNOWN_CONTENT_LENGTH);
  }

  public NettyFeedableHttpEntity(long contentLength) {
    BlockingBidirectionalStream bidirectionalStream = new BlockingBidirectionalStream();
    this.asInputStream = bidirectionalStream.getInputStream();
    this.dataSink = bidirectionalStream.getOutputStream();
    this.contentLength = contentLength;
    this.futureTrailers = new CompletableFuture<>();
  }

  @Override
  public boolean isStreaming() {
    return true;
  }

  @Override
  public InputStream getContent() {
    return asInputStream;
  }

  @Override
  public byte[] getBytes() throws IOException {
    return toByteArray(asInputStream);
  }

  @Override
  public Optional<Long> getLength() {
    return contentLength == UNKNOWN_CONTENT_LENGTH ? Optional.empty() : Optional.of(contentLength);
  }

  @Override
  public OptionalLong getBytesLength() {
    return contentLength == UNKNOWN_CONTENT_LENGTH ? OptionalLong.empty() : OptionalLong.of(contentLength);
  }

  @Override
  public void feed(ByteBuffer data) throws IOException {
    byte[] bytes = new byte[data.remaining()];
    data.get(bytes);

    LOGGER.debug("Feeding entity with {} bytes", bytes.length);
    dataSink.write(bytes, 0, bytes.length);
  }

  @Override
  public void error(Exception exception) {
    LOGGER.debug("Marking entity with error", exception);
    dataSink.cancel(exception);
  }

  @Override
  public void complete() throws IOException {
    completeWithTrailers(emptyMultiMap());
  }

  @Override
  public synchronized void completeWithTrailers(MultiMap<String, String> trailers) throws IOException {
    LOGGER.atDebug().log(() -> "Completing HTTP entity with {} trailers" + trailers.size());
    futureTrailers.complete(trailers);
    dataSink.close();
  }

  @Override
  public void onComplete(BiConsumer<? super MultiMap<String, String>, ? super Throwable> completionCallback) {
    LOGGER.debug("Registering a callback for the entity completion");
    futureTrailers.whenComplete(completionCallback);
  }
}
