/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelPromise;

/**
 * Strategy to write parts of the http responses in parts.
 */
public sealed interface HttpWriter permits Http1Writer, Http2Writer {

  /**
   * Sends the header part of the response.
   *
   * @param request  the request we're responding.
   * @param response the response being sent.
   * @param promise  promise to be called when write is done or fails.
   */
  void writeResponseHeader(HttpRequest request, HttpResponse response, ChannelPromise promise);

  /**
   * Writes partial content.
   *
   * @param content partial content to send.
   * @param isLast  whether it's the last part of the response content.
   * @param promise promise to be called when write is done or fails.
   */
  void writeContent(ByteBuf content, boolean isLast, ChannelPromise promise);

  /**
   * Writes HTTP Trailers. Only supported for HTTP/2.
   *
   * @param trailers trailers to send.
   * @param promise  promise to be called when write is done or fails.
   */
  void writeTrailers(MultiMap<String, String> trailers, ChannelPromise promise);
}
