/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.http.api.domain.HttpProtocol.HTTP_2;
import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.decodeQueryString;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.extractQueryParams;
import static org.mule.runtime.http.api.utils.UriCache.getUriFromString;
import static org.mule.service.http.netty.impl.util.NettyUtils.toNioBuffer;

import static io.netty.buffer.ByteBufUtil.getBytes;

import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;
import org.mule.runtime.http.api.domain.HttpProtocol;
import org.mule.runtime.http.api.domain.entity.FeedableHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.service.http.netty.impl.message.content.NettyFeedableHttpEntity;
import org.mule.service.http.netty.impl.util.NettyUtils;

import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2HeadersFrame;

/**
 * Adapter from the Netty's abstraction of request frames to Mule's HTTP/2 version of {@link HttpRequest}.
 */
public class NettyHttp2RequestAdapter implements HttpRequest {

  private static final char PSEUDO_HEADER_PREFIX = ':';

  private final Http2HeadersFrame headersFrame;

  private final URI uri;
  private final String path;
  private final MultiMap<String, String> queryParams;

  private final LazyValue<MultiMap<String, String>> headersAsMultiMap;
  private final MultiMap<String, String> trailers = new CaseInsensitiveMultiMap(!PRESERVE_HEADER_CASE);

  private final FeedableHttpEntity entity;

  public NettyHttp2RequestAdapter(Http2HeadersFrame headersFrame) {
    this.headersFrame = headersFrame;

    String uriString = headersFrame.headers().path().toString();
    this.uri = getUriFromString(uriString);
    this.path = uri.getPath();
    this.queryParams = decodeQueryString(extractQueryParams(uriString));

    this.entity = new NettyFeedableHttpEntity();

    this.headersAsMultiMap = new LazyValue<>(() -> headersToMultiMap(headersFrame.headers()));
  }

  @Override
  public HttpProtocol getProtocol() {
    return HTTP_2;
  }

  @Override
  public String getPath() {
    return path;
  }

  @Override
  public String getMethod() {
    return headersFrame.headers().method().toString();
  }

  @Override
  public URI getUri() {
    return uri;
  }

  @Override
  public MultiMap<String, String> getQueryParams() {
    return queryParams;
  }

  @Override
  public HttpEntity getEntity() {
    return entity;
  }

  @Override
  public Collection<String> getHeaderNames() {
    return headersAsMultiMap.get().keySet();
  }

  @Override
  public boolean containsHeader(String headerName) {
    return headersFrame.headers().contains(headerName);
  }

  @Override
  public String getHeaderValue(String headerName) {
    return getHeaderValueIgnoreCase(headerName);
  }

  @Override
  public String getHeaderValueIgnoreCase(String headerName) {
    if (headersAsMultiMap.isComputed()) {
      return headersAsMultiMap.get().get(headerName);
    }

    CharSequence asCharSequence = headersFrame.headers().get(headerName);
    return asCharSequence == null ? null : asCharSequence.toString();
  }

  @Override
  public Collection<String> getHeaderValues(String headerName) {
    return getHeaderValuesIgnoreCase(headerName);
  }

  @Override
  public Collection<String> getHeaderValuesIgnoreCase(String headerName) {
    if (headersAsMultiMap.isComputed()) {
      return headersAsMultiMap.get().getAll(headerName);
    }

    List<CharSequence> values = headersFrame.headers().getAll(headerName);
    Collection<String> asStrings = new LinkedList<>();
    for (CharSequence value : values) {
      asStrings.add(value.toString());
    }
    return asStrings;
  }

  @Override
  public MultiMap<String, String> getHeaders() {
    return headersAsMultiMap.get();
  }

  private static MultiMap<String, String> headersToMultiMap(Http2Headers nettyHeaders) {
    var asMultiMap = new CaseInsensitiveMultiMap(!PRESERVE_HEADER_CASE);
    for (var entry : nettyHeaders) {
      var headerName = entry.getKey();
      if (headerName.charAt(0) != PSEUDO_HEADER_PREFIX) {
        var headerValue = entry.getValue();
        asMultiMap.put(headerName.toString(), headerValue.toString());
      }
    }
    return asMultiMap;
  }

  public void finishEntityAndSetTrailers() throws IOException {
    entity.completeWithTrailers(trailers);
  }

  public void pushContent(ByteBuf content) throws IOException {
    entity.feed(toNioBuffer(content));
  }

  public void addTrailer(String name, String value) {
    trailers.put(name, value);
  }

}
