/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.trailersAsFuture;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase.IgnoreResponseStatusCallback;

import java.io.IOException;

class EchoLikeRequestHandler implements RequestHandler {

  @Override
  public void handleRequest(HttpRequestContext reqCtx, HttpResponseReadyCallback callback) {
    var response = generateResponse(reqCtx);
    callback.responseReady(response, new IgnoreResponseStatusCallback());
  }

  private HttpResponse generateResponse(HttpRequestContext reqCtx) {
    try {
      var echoBody = generateResponseBody(reqCtx);
      return HttpResponse.builder()
          .statusCode(200)
          .entity(echoBody)
          .build();
    } catch (Exception e) {
      return HttpResponse.builder()
          .statusCode(500)
          .entity(new StringHttpEntity("Error: " + e.getMessage()))
          .build();
    }
  }

  private HttpEntity generateResponseBody(HttpRequestContext reqCtx) throws Exception {
    var req = reqCtx.getRequest();
    String body = extractBodyAsString(req);
    String queryParams = req.getQueryParams().toString();
    String uriString = req.getUri().toString();
    var trailers = trailersAsFuture(req.getEntity()).get();
    return new StringHttpEntity("""
        Request{
          body: %s,
          uri: %s,
          queryParams: %s
          trailers: %s
        }
        """.formatted(body, uriString, queryParams, trailers.toString()));
  }

  // This method consumes the full body SYNCHRONOUSLY.
  private String extractBodyAsString(HttpRequest request) {
    try {
      return new String(request.getEntity().getBytes());
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
