/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.trailersAsFuture;
import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_TRAILERS;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.api.util.MultiMap.StringMultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;

import io.qameta.allure.Story;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Story(HTTP_2_TRAILERS)
class Http2TrailersTestCase extends AbstractHttp2ClientServerTestCase {

  private static final String ECHO_PATH = "/echo";
  private static final String QUERY_PARAMS_TO_TRAILERS_PATH = "/query-params-to-trailers";

  private static final String TRAILER_NAME = "the-trailer";
  private static final String TRAILER_CLIENT_TO_SERVER = "client-to-server";
  private static final String TRAILER_SERVER_TO_CLIENT = "server-to-client";

  public Http2TrailersTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void initRequestHandler() {
    // The echo handler will parse the client-to-server trailers and stringify them in the response body.
    server.addRequestHandler(ECHO_PATH, new EchoLikeRequestHandler()).start();

    // Add a handler that responds with trailers copied from the input query parameters, to test server-to-client trailers.
    server.addRequestHandler(QUERY_PARAMS_TO_TRAILERS_PATH, ((requestContext, responseCallback) -> {
      // To send trailers, you need a feedable entity
      var feedableEntity = service.getEntityFactory().feedable();

      // And set that feedable as the entity of the response
      var response = HttpResponse.builder()
          .statusCode(200)
          .entity(feedableEntity)
          .build();

      // Now you have a response, you can start the response
      responseCallback.responseReady(response, new IgnoreResponseStatusCallback());

      try {
        // And when you have the trailers calculated (in this case a copy of the query params), complete the feedable.
        feedableEntity.completeWithTrailers(requestContext.getRequest().getQueryParams().toImmutableMultiMap());
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    })).start();
  }

  @Test
  void trailersClientToServer() throws Exception {
    // You need a feedable entity to send trailers.
    var feedableEntity = service.getEntityFactory().feedable();

    // A normal request, with the feedable entity
    var request = HttpRequest.builder()
        .uri(urlForPath(server, ECHO_PATH))
        .entity(feedableEntity)
        .build();
    var responseFuture = client.sendAsync(request);

    // When you have the trailers ready, you complete the feedable with them
    feedableEntity.completeWithTrailers(trailersFrom(TRAILER_NAME, TRAILER_CLIENT_TO_SERVER));

    // And here we're asserting that the echo handler successfully received the trailers and added them to the response body
    var response = responseFuture.get();
    assertThat(response.getStatusCode(), is(200));
    String bodyText = new String(response.getEntity().getBytes());
    assertThat(bodyText, containsString(multimapStringFor(TRAILER_NAME, TRAILER_CLIENT_TO_SERVER)));
  }

  @Test
  void trailersServerToClient() throws Exception {
    // We're just adding some query parameters here, the magic will happen in the request handler, which is sending the trailers
    // from server to client...
    var request = HttpRequest.builder()
        .uri(urlForPath(server, QUERY_PARAMS_TO_TRAILERS_PATH + "?" + TRAILER_NAME + "=" + TRAILER_SERVER_TO_CLIENT))
        .build();
    var response = client.send(request);
    assertThat(response.getStatusCode(), is(200));

    // If the server's request handler wrote the trailers correctly, they should be received here...
    var responseTrailers = trailersAsFuture(response.getEntity()).get();
    assertThat(responseTrailers.get(TRAILER_NAME), is(TRAILER_SERVER_TO_CLIENT));
  }

  private static MultiMap<String, String> trailersFrom(String name, String value) {
    var reqTrailers = new StringMultiMap();
    reqTrailers.put(name, value);
    return reqTrailers;
  }

  private static String multimapStringFor(String name, String value) {
    return "trailers: MultiMap{[%s=[%s]]}".formatted(name, value);
  }
}
