/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.service.http.test.netty.AllureConstants.HTTP_2;
import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_TRAILERS;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;

import org.mule.service.http.netty.impl.server.NettyHttp2RequestAdapter;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import io.netty.handler.codec.http2.DefaultHttp2Headers;
import io.netty.handler.codec.http2.DefaultHttp2HeadersFrame;
import io.netty.handler.codec.http2.Http2HeadersFrame;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Feature(HTTP_2)
@Story(HTTP_2_TRAILERS)
class NettyHttp2RequestAdapterTestCase extends AbstractHttpTestCase {

  private Http2HeadersFrame headersFrame;

  @BeforeEach
  void setUp() {
    var headers = new DefaultHttp2Headers();
    headers.method("POST");
    headers.path("/some/path?queryParam=value");
    headers.scheme("http");
    headers.add("content-length", "0");
    headers.add("x-header1", "value");
    headers.add("x-header2", List.of("value1", "value2"));
    headersFrame = new DefaultHttp2HeadersFrame(headers);
  }

  @Test
  void methodComesFromThePseudoHeaders() {
    var muleResponse = new NettyHttp2RequestAdapter(headersFrame);
    assertThat(muleResponse.getMethod(), is("POST"));
  }

  @Test
  void pathComesFromThePseudoHeaders() {
    var muleResponse = new NettyHttp2RequestAdapter(headersFrame);
    assertThat(muleResponse.getPath(), is("/some/path"));
  }

  @Test
  void queryParametersComesFromThePseudoHeaders() {
    var muleResponse = new NettyHttp2RequestAdapter(headersFrame);
    var queryParams = muleResponse.getQueryParams();
    assertThat(queryParams.get("queryParam"), is("value"));
  }

  @Test
  void getHeadersIndividually() {
    var muleResponse = new NettyHttp2RequestAdapter(headersFrame);
    assertThat(muleResponse.getHeaderValue("content-length"), is("0"));
    assertThat(muleResponse.getHeaderValue("x-header1"), is("value"));
    assertThat(muleResponse.getHeaderValues("x-header2"), contains("value1", "value2"));
  }

  @Test
  void getAllHeaderNames() {
    var muleResponse = new NettyHttp2RequestAdapter(headersFrame);
    assertThat(muleResponse.getHeaderNames(), contains("content-length", "x-header1", "x-header2"));
  }

  @Test
  void getAllHeadersAsMultiMap() {
    var muleResponse = new NettyHttp2RequestAdapter(headersFrame);
    var headers = muleResponse.getHeaders();
    assertThat(headers.get("content-length"), is("0"));
    assertThat(headers.get("X-Header1"), is("value"));
    assertThat(headers.getAll("x-header2"), contains("value1", "value2"));
  }

  @Test
  void responseWithTrailers() throws IOException, ExecutionException, InterruptedException {
    var muleResponse = new NettyHttp2RequestAdapter(headersFrame);

    var trailer1 = new CompletableFuture<String>();
    var trailer2 = new CompletableFuture<String>();
    muleResponse.getEntity().onComplete((trailers, throwable) -> {
      trailer1.complete(trailers.get("x-trailer1"));
      trailer2.complete(trailers.get("x-trailer2"));
    });

    muleResponse.addTrailer("x-trailer1", "value1");
    muleResponse.addTrailer("x-trailer2", "value2");
    muleResponse.finishEntityAndSetTrailers();

    assertThat(trailer1.get(), is("value1"));
    assertThat(trailer2.get(), is("value2"));
  }
}
