/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message.content;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;
import static org.mule.service.http.netty.impl.server.util.HttpParser.parseMultipartContent;

import static java.util.Optional.ofNullable;
import static java.util.concurrent.CompletableFuture.completedFuture;

import static org.apache.commons.io.IOUtils.toByteArray;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;

/**
 * Represents a received multipart body, which can be accessed raw or parsed.
 *
 * @since 1.0
 */
public class StreamedMultipartHttpEntity implements HttpEntity {

  private final InputStream content;
  private final String contentType;
  private final Long contentLength;
  private final CompletableFuture<MultiMap<String, String>> futureTrailers;

  public StreamedMultipartHttpEntity(InputStream content, String contentType) {
    this(content, contentType, null, completedFuture(emptyMultiMap()));
  }

  public StreamedMultipartHttpEntity(InputStream content, String contentType,
                                     CompletableFuture<MultiMap<String, String>> futureTrailers) {
    this(content, contentType, null, futureTrailers);
  }

  public StreamedMultipartHttpEntity(InputStream content, String contentType, Long contentLength) {
    this(content, contentType, contentLength, completedFuture(emptyMultiMap()));
  }

  public StreamedMultipartHttpEntity(InputStream content, String contentType, Long contentLength,
                                     CompletableFuture<MultiMap<String, String>> futureTrailers) {
    this.content = content;
    this.contentType = contentType;
    this.contentLength = contentLength;
    this.futureTrailers = futureTrailers;
  }

  @Override
  public boolean isStreaming() {
    return true;
  }

  @Override
  public boolean isComposed() {
    return true;
  }

  @Override
  public InputStream getContent() throws UnsupportedOperationException {
    return content;
  }

  @Override
  public byte[] getBytes() throws UnsupportedOperationException {
    try {
      return toByteArray(content);
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public Collection<HttpPart> getParts() throws IOException, UnsupportedOperationException {
    return parseMultipartContent(content, contentType);
  }

  @Override
  public Optional<Long> getLength() {
    return ofNullable(contentLength);
  }


  @Override
  public void onComplete(BiConsumer<? super MultiMap<String, String>, ? super Throwable> completionCallback) {
    futureTrailers.whenComplete(completionCallback);
  }
}
