/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.service.http.netty.impl.server.util.DefaultServerAddress;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.codec.http2.Http2HeadersFrame;
import io.netty.handler.ssl.SslHandler;
import org.slf4j.Logger;

/**
 * It receives the HTTP/2 frames and uses them to create the corresponding {@link HttpRequest} instances. Then it delegates the
 * built objects to the {@link HttpListenerRegistry} to be handled.
 */
public class NettyToMuleHttp2RequestHandlerAdapter extends SimpleChannelInboundHandler<Object> {

  private static final Logger LOGGER = getLogger(NettyToMuleHttp2RequestHandlerAdapter.class);

  private final HttpListenerRegistry httpListenerRegistry;
  private final SslHandler sslHandler;
  private final Executor ioExecutor;

  private NettyHttp2RequestAdapter request;

  public NettyToMuleHttp2RequestHandlerAdapter(HttpListenerRegistry httpListenerRegistry, SslHandler sslHandler,
                                               Executor ioExecutor) {
    this.httpListenerRegistry = httpListenerRegistry;
    this.sslHandler = sslHandler;
    this.ioExecutor = ioExecutor;
  }

  @Override
  protected void channelRead0(ChannelHandlerContext ctx, Object msg) throws Exception {
    if (msg instanceof Http2DataFrame dataFrame) {
      onDataFrameRead(dataFrame);
    } else if (msg instanceof Http2HeadersFrame headersFrame) {
      if (request == null) {
        // This is a request headers frame
        onHeadersFrameRead(ctx, headersFrame);
      } else {
        // This is a trailers frame
        onTrailersFrameRead(headersFrame);
      }
    } else {
      super.channelRead(ctx, msg);
    }
  }

  private void onHeadersFrameRead(ChannelHandlerContext ctx, Http2HeadersFrame headersFrame) throws IOException {
    InetSocketAddress socketAddress = (InetSocketAddress) ctx.channel().localAddress();
    DefaultServerAddress serverAddress = new DefaultServerAddress(socketAddress.getAddress(), socketAddress.getPort());

    request = new NettyHttp2RequestAdapter(headersFrame);

    if (headersFrame.isEndStream()) {
      request.finishEntityAndSetTrailers();
    }

    var requestHandler = httpListenerRegistry.getRequestHandler(serverAddress, request);
    var requestContext = new NettyHttpRequestContext(request, ctx, sslHandler);
    var responseReadyCallback = new NettyHttp2RequestReadyCallback(ctx, headersFrame.stream(), request, ioExecutor);
    try {
      ioExecutor.execute(() -> requestHandler.handleRequest(requestContext, responseReadyCallback));
    } catch (RejectedExecutionException ree) {
      LOGGER.debug("IO Executor is busy, processing request synchronously");
      requestHandler.handleRequest(requestContext, responseReadyCallback);
    }
  }

  private void onTrailersFrameRead(Http2HeadersFrame trailersFrame) throws IOException {
    // Process trailer headers
    for (var entry : trailersFrame.headers()) {
      var headerName = entry.getKey().toString();
      var headerValue = entry.getValue().toString();
      request.addTrailer(headerName, headerValue);
    }

    if (trailersFrame.isEndStream()) {
      request.finishEntityAndSetTrailers();
    }
  }

  private void onDataFrameRead(Http2DataFrame dataFrame) throws IOException {
    ByteBuf content = dataFrame.content();
    request.pushContent(content);
    if (dataFrame.isEndStream()) {
      request.finishEntityAndSetTrailers();
    }
  }
}
