/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.service.http.netty.impl.server.KeepAliveHandler.TIMEOUT_READING_REQUEST;

import static io.netty.handler.codec.http.HttpResponseStatus.REQUEST_TIMEOUT;
import static io.netty.handler.timeout.IdleState.ALL_IDLE;
import static io.netty.handler.timeout.IdleState.READER_IDLE;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.socket.ChannelInputShutdownEvent;
import io.netty.handler.codec.http2.DefaultHttp2DataFrame;
import io.netty.handler.codec.http2.DefaultHttp2Headers;
import io.netty.handler.codec.http2.DefaultHttp2HeadersFrame;
import io.netty.handler.codec.http2.Http2ChannelDuplexHandler;
import io.netty.handler.codec.http2.Http2ConnectionHandler;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.codec.http2.Http2Exception;
import io.netty.handler.codec.http2.Http2FrameStream;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2HeadersFrame;
import io.netty.handler.timeout.IdleStateEvent;

/**
 * Handler to close server-side connections when needed.
 * <p>
 * It's mainly based in Netty's built-in {@link io.netty.handler.codec.http.HttpServerKeepAliveHandler}, but adapted for HTTP2
 */
public class Http2ConnectionCleanupHandler extends Http2ChannelDuplexHandler {

  private final Http2ConnectionHandler connectionHandler;

  public Http2ConnectionCleanupHandler(Http2ConnectionHandler connectionHandler) {
    this.connectionHandler = connectionHandler;
  }

  @Override
  public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Http2Exception {
    if (evt instanceof IdleStateEvent idleStateEvent) {
      int numActiveStreams = connectionHandler.connection().numActiveStreams();
      if (READER_IDLE == idleStateEvent.state() && numActiveStreams > 0) {
        // The reader idle timeout should only apply if there is an inflight request
        forEachActiveStream(frameStream -> {
          // Because of multiplexing, we need to send the timeout response on each stream
          sendTimeoutResponse(ctx, frameStream);
          return true;
        });
        return;
      }
      if (ALL_IDLE == idleStateEvent.state() && numActiveStreams == 0) {
        // The connection idle timeout should only apply if there is NOT an inflight request
        ctx.close();
        return;
      }
    }
    if (evt instanceof ChannelInputShutdownEvent) {
      ctx.close();
      return;
    }
    ctx.fireUserEventTriggered(evt);
  }

  private void sendTimeoutResponse(ChannelHandlerContext ctx, Http2FrameStream frameStream) {
    Http2Headers headers = new DefaultHttp2Headers().status(String.valueOf(REQUEST_TIMEOUT.code()));
    Http2HeadersFrame rejection = new DefaultHttp2HeadersFrame(headers, false).stream(frameStream);
    ctx.writeAndFlush(rejection);

    ByteBuf buffer = ctx.alloc().buffer();
    buffer.writeBytes(TIMEOUT_READING_REQUEST.getBytes());
    Http2DataFrame dataFrame = new DefaultHttp2DataFrame(buffer, true).stream(frameStream);
    ctx.writeAndFlush(dataFrame).addListener(ChannelFutureListener.CLOSE);
  }
}
