/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static org.mule.runtime.api.metadata.MediaType.MULTIPART_MIXED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NOT_MODIFIED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.NO_CONTENT;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.RESET_CONTENT;

import static java.lang.Long.parseLong;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.service.http.netty.impl.message.content.StreamedMultipartHttpEntity;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;

import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpResponse;
import reactor.netty.http.client.HttpClientResponse;

public class HttpResponseCreatorUtils {

  /**
   * Creates the most convenient implementation of {@link HttpEntity} depending on the known parameters.
   *
   * @param stream         the body as {@link InputStream}.
   * @param contentType    the content type.
   * @param contentLength  the "Content-Length" header value.
   * @param statusCode     the status code.
   * @param futureTrailers the trailers, as a {@link CompletableFuture}.
   * @return an {@link HttpEntity}.
   */
  public static HttpEntity createEntity(InputStream stream,
                                        String contentType,
                                        String contentLength,
                                        int statusCode,
                                        CompletableFuture<MultiMap<String, String>> futureTrailers) {
    long contentLengthAsLong = parseContentLength(contentLength);

    if (isMultipartMixed(contentType)) {
      return createMultipartEntity(stream, contentLengthAsLong, contentType, futureTrailers);
    }

    if (contentLengthAsLong > 0) {
      return new InputStreamHttpEntity(stream, contentLengthAsLong, futureTrailers);
    }

    if (contentLengthAsLong == 0 || isEmptyResponse(statusCode)) {
      return new EmptyHttpEntity(futureTrailers);
    }

    return new InputStreamHttpEntity(stream, futureTrailers);
  }

  /**
   * Extracts a header from the Netty's abstraction of HTTP Response.
   *
   * @param response   the Netty's abstraction of HTTP Response.
   * @param headerName the header name.
   * @return the value, if any, of the header with the passed name, or null if the header is not present.
   */
  public String extractHeader(HttpResponse response, String headerName) {
    HttpHeaders headers = response.headers();
    return headers.contains(headerName) ? headers.get(headerName) : null;
  }

  /**
   * Extracts a header from the Reactor Netty's abstraction of HTTP Response.
   *
   * @param response   the Reactor Netty's abstraction of HTTP Response.
   * @param headerName the header name.
   * @return the value, if any, of the header with the passed name, or null if the header is not present.
   */
  public String extractHeader(HttpClientResponse response, String headerName) {
    HttpHeaders headers = response.responseHeaders();
    return headers.get(headerName);
  }

  /**
   * Creates a Mule's {@link HttpResponseBuilder} from the Netty's abstraction {@link HttpResponse}
   *
   * @param response Netty's abstraction {@link HttpResponse}.
   * @return a new {@link HttpResponseBuilder}
   */
  public HttpResponseBuilder buildResponseHeaders(HttpResponse response) {
    HttpResponseBuilder responseBuilder =
        org.mule.runtime.http.api.domain.message.response.HttpResponse.builder().statusCode(response.status().code())
            .reasonPhrase(response.status().reasonPhrase());

    HttpHeaders headers = response.headers();
    if (!headers.isEmpty()) {
      headers.names().forEach(headerName -> responseBuilder.addHeaders(headerName, headers.getAll(headerName)));
    }
    return responseBuilder;
  }

  /**
   * Extracts the trailers from the Reactor Netty's abstraction of response as a {@link CompletableFuture}.
   *
   * @param response Reactor Netty's {@link HttpClientResponse}.
   * @return a {@link CompletableFuture} with the trailers.
   */
  public static CompletableFuture<MultiMap<String, String>> trailersAsFuture(HttpClientResponse response) {
    var trailersFuture = new CompletableFuture<MultiMap<String, String>>();
    response.trailerHeaders().toFuture().whenComplete((trailers, throwable) -> {
      if (throwable != null) {
        trailersFuture.completeExceptionally(throwable);
      } else {
        var asMultiMap = new MultiMap.StringMultiMap();
        for (var trailer : trailers) {
          asMultiMap.put(trailer.getKey(), trailer.getValue());
        }
        trailersFuture.complete(asMultiMap);
      }
    });
    return trailersFuture;
  }

  /**
   * Extracts the trailers from the Mule's abstraction of entity.
   *
   * @param httpEntity Mule's {@link HttpEntity}.
   * @return a {@link CompletableFuture} with the trailers.
   */
  public static CompletableFuture<MultiMap<String, String>> trailersAsFuture(HttpEntity httpEntity) {
    var completableFuture = new CompletableFuture<MultiMap<String, String>>();
    httpEntity.onComplete((entity, throwable) -> {
      if (throwable != null) {
        completableFuture.completeExceptionally(throwable);
      } else {
        completableFuture.complete(entity);
      }
    });

    return completableFuture;
  }

  private static long parseContentLength(String contentLength) {
    return contentLength != null ? parseLong(contentLength) : -1L;
  }

  private static boolean isMultipartMixed(String contentType) {
    return contentType != null && contentType.startsWith(MULTIPART_MIXED.getPrimaryType());
  }

  private static HttpEntity createMultipartEntity(InputStream stream, long contentLengthAsLong, String contentType,
                                                  CompletableFuture<MultiMap<String, String>> futureTrailers) {
    return contentLengthAsLong >= 0 ? new StreamedMultipartHttpEntity(stream, contentType, contentLengthAsLong, futureTrailers)
        : new StreamedMultipartHttpEntity(stream, contentType, futureTrailers);
  }

  private static boolean isEmptyResponse(int statusCode) {
    return statusCode == NO_CONTENT.getStatusCode() || statusCode == NOT_MODIFIED.getStatusCode()
        || statusCode == RESET_CONTENT.getStatusCode();
  }
}
