/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.buildResponseHeaders;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.createEntity;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.createMultipartEntity;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.extractHeader;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.isEmptyResponseStatus;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.isMultipartMixed;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.parseContentLength;
import static org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils.trailersAsFuture;

import static java.util.concurrent.CompletableFuture.completedFuture;

import static io.netty.buffer.ByteBufUtil.getBytes;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.service.http.netty.impl.message.content.NettyFeedableHttpEntity;
import org.mule.service.http.netty.impl.message.content.OnlyTrailersFeedable;
import org.mule.service.http.netty.impl.message.content.StreamAndFutureFeedable;
import org.mule.service.http.netty.impl.streaming.BlockingBidirectionalStream;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.http.HttpResponse;
import reactor.netty.http.client.HttpClientResponse;

public class HttpResponseCreator {

  private static final String HEADER_CONTENT_TYPE = CONTENT_TYPE.toLowerCase();
  private static final String HEADER_CONTENT_LENGTH = CONTENT_LENGTH.toLowerCase();
  public static final byte[] EMPTY_BYTES = new byte[0];

  public org.mule.runtime.http.api.domain.message.response.HttpResponse create(HttpResponse response, InputStream inputStream) {
    HttpResponseBuilder responseBuilder = buildResponseHeaders(response);
    String contentType = extractHeader(response, HEADER_CONTENT_TYPE);
    String contentLengthAsString = extractHeader(response, HEADER_CONTENT_LENGTH);
    long contentLength = parseContentLength(contentLengthAsString);
    responseBuilder.entity(createEntity(inputStream, contentType, contentLength,
                                        response.status().code(), completedFuture(emptyMultiMap())));

    return responseBuilder.build();
  }

  public org.mule.runtime.http.api.domain.message.response.HttpResponse create(HttpClientResponse response, ByteBuf byteBuf) {
    byte[] bytes = byteBuf == null ? EMPTY_BYTES : getBytes(byteBuf);
    InputStream inputStream = byteBuf == null ? null : new ByteArrayInputStream(bytes);

    int statusCode = response.status().code();
    String contentType = extractHeader(response, HEADER_CONTENT_TYPE);
    var trailersFuture = trailersAsFuture(response);
    HttpEntity entity = createEntity(inputStream, contentType, bytes.length, statusCode, trailersFuture);
    return new ReactorNettyResponseWrapper(response, entity, new OnlyTrailersFeedable(trailersFuture));
  }

  public ReactorNettyResponseWrapper createFeedable(HttpClientResponse response) {
    int statusCode = response.status().code();
    String contentType = extractHeader(response, HEADER_CONTENT_TYPE);
    String contentLength = extractHeader(response, HEADER_CONTENT_LENGTH);
    long contentLengthAsLong = parseContentLength(contentLength);

    if (isMultipartMixed(contentType)) {
      // Multipart entities will use a BlockingBidirectionalStream.
      var futureTrailers = trailersAsFuture(response);
      var bidirectionalStream = new BlockingBidirectionalStream();
      var dataSink = bidirectionalStream.getOutputStream();
      var entity = createMultipartEntity(bidirectionalStream.getInputStream(), contentLengthAsLong, contentType, futureTrailers);
      return new ReactorNettyResponseWrapper(response, entity, new StreamAndFutureFeedable(dataSink, futureTrailers));
    }

    if (contentLengthAsLong == 0 || isEmptyResponseStatus(statusCode)) {
      var futureTrailers = trailersAsFuture(response);
      var entity = new EmptyHttpEntity(futureTrailers);
      return new ReactorNettyResponseWrapper(response, entity, new OnlyTrailersFeedable(futureTrailers));
    }

    var entity = new NettyFeedableHttpEntity(contentLengthAsLong);
    return new ReactorNettyResponseWrapper(response, entity, entity);
  }
}
