/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.benchmark;

import static java.lang.Integer.getInteger;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.anyUrl;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.core.WireMockConfiguration.wireMockConfig;
import static org.openjdk.jmh.annotations.Mode.Throughput;

import org.mule.service.http.netty.impl.message.HttpResponseCreator;

import com.github.tomakehurst.wiremock.WireMockServer;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import reactor.netty.http.client.HttpClient;
import reactor.netty.http.client.HttpClientResponse;

@BenchmarkMode(Throughput)
@OutputTimeUnit(MILLISECONDS)
@Fork(value = 1)
public class HttpResponseCreatorBenchmark {

  @State(Scope.Benchmark)
  public static class ExecPlan {

    public HttpResponseCreator httpResponseCreator = new HttpResponseCreator();
    public HttpClientResponse reactorNettyResponse;

    public ByteBuf body = Unpooled.EMPTY_BUFFER;

    @Setup
    public void setUp() {
      Integer port = getInteger("mock.server.port");

      WireMockServer mockServer = new WireMockServer(wireMockConfig().port(port));
      mockServer.start();
      mockServer.stubFor(get(anyUrl())
          .willReturn(aResponse()
              .withStatus(200)
              .withHeader("Content-Length", "0")
              .withHeader("X-Header1", "value")
              .withHeader("X-Header2", "value")
              .withHeader("X-Header3", "value")
              .withHeader("X-Header4", "value")
              .withHeader("X-Header5", "value")));

      HttpClient reactorNettyClient = HttpClient.create();
      reactorNettyResponse = reactorNettyClient
          .get().uri("http://localhost:" + port + "/")
          .response()
          .block();

      httpResponseCreator = new HttpResponseCreator();

      mockServer.stop();
    }
  }

  @Benchmark
  @Threads(1)
  @Warmup(iterations = 3, time = 3, timeUnit = SECONDS)
  @Measurement(iterations = 10, time = 5, timeUnit = SECONDS)
  public void adaptReactorNettyResponseToMule(ExecPlan state) {
    state.httpResponseCreator.create(state.reactorNettyResponse, state.body);
  }
}
