/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import static org.mule.service.http.netty.impl.util.MuleToNettyUtils.addAllRequestHeaders;

import static io.netty.handler.codec.http.HttpResponseStatus.UNAUTHORIZED;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.netty.impl.client.ReactorNettyClient;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.http.DefaultHttpHeaders;
import reactor.core.publisher.Flux;
import reactor.netty.http.client.HttpClientResponse;

/**
 * Response handler that handles Authentication, including Basic and Digest Whenever a response returns a 401 (Unauthorized), it
 * checks whether it needs to emit new headers and retries the request with the new header
 */
public class AuthenticationHandler {

  private final ReactorNettyClient httpClient;
  private final AuthenticationEngine authEngine;

  public AuthenticationHandler(ReactorNettyClient client, AuthenticationEngine authHeadersProvider) {
    this.httpClient = client;
    this.authEngine = authHeadersProvider;
  }

  public boolean needsAuth(HttpClientResponse httpResponse, HttpRequestOptions options) {
    Optional<HttpAuthentication> authentication = options.getAuthentication();
    return authentication.isPresent() && !authEngine.hasFinished() && httpResponse.status().code() == UNAUTHORIZED.code();
  }

  public Flux<ByteBuf> doHandle(HttpRequest request, HttpRequestOptions requestOptions, HttpClientResponse httpResponse,
                                CompletableFuture<HttpResponse> result) {
    Optional<HttpAuthentication> optionalHttpAuthentication = requestOptions.getAuthentication();
    if (!optionalHttpAuthentication.isPresent()) {
      throw new MuleRuntimeException(createStaticMessage("Authentication should be configured to call this method"));
    }
    return httpClient.sendAsyncRequest(request, requestOptions,
                                       getAllHeaders(request, httpResponse, authEngine),
                                       (response, content) -> {
                                         if (needsAuth(response, requestOptions)) {
                                           return doHandle(request, requestOptions, response, result);
                                         } else {
                                           return httpClient.receiveContent(response, content, result);
                                         }
                                       }, result);
  }

  private io.netty.handler.codec.http.HttpHeaders getAllHeaders(HttpRequest request, HttpClientResponse httpResponse,
                                                                AuthenticationEngine authenticationHeadersProvider) {
    io.netty.handler.codec.http.HttpHeaders httpHeaders = new DefaultHttpHeaders();
    addAllRequestHeaders(request, httpHeaders, request.getUri());
    if (httpResponse.responseHeaders().contains(WWW_AUTHENTICATE)) {
      // If the header has many values, select the one matching the configured authentication
      httpHeaders.add(WWW_AUTHENTICATE, httpResponse.responseHeaders().getAll(WWW_AUTHENTICATE)
          .stream()
          .filter(authenticationHeadersProvider.getType().name()::equalsIgnoreCase)
          .findAny()
          .orElseGet(() -> httpResponse.responseHeaders().get(WWW_AUTHENTICATE)));
    }
    httpHeaders
        .add(authenticationHeadersProvider.getAuthHeaders(httpHeaders));
    return httpHeaders;
  }
}
