/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.functional.junit4.matchers.ThrowableMessageMatcher.hasMessage;
import static org.mule.runtime.http.api.domain.message.request.HttpRequest.builder;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.junit.internal.matchers.ThrowableCauseMatcher.hasCause;
import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Issue;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Issue("W-19068996")
class SimpleClientTestCase extends AbstractHttpServiceTestCase {

  private HttpClient httpClient;

  public SimpleClientTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setUp() {
    var config = new HttpClientConfiguration.Builder().setName("HTTP/2 Client").build();
    httpClient = service.getClientFactory().create(config);
    httpClient.start();
  }

  @AfterEach
  void tearDown() {
    httpClient.stop();
  }

  @Test
  void clientChecksThatUriHasScheme() throws URISyntaxException {
    testIncorrectUri("/theUri", "The uri provided '/theUri' must contain a scheme.");
  }

  @Test
  void clientChecksThatUriHasHost() throws URISyntaxException {
    testIncorrectUri("https:///theUri", "The uri provided 'https:///theUri' must contain a host.");
  }

  private void testIncorrectUri(String uriString, String expectedMessage) throws URISyntaxException {
    var uriWithoutHost = new URI(uriString);
    var future = httpClient.sendAsync(builder().uri(uriWithoutHost).build());
    var error = assertThrows(ExecutionException.class, future::get);
    assertThat(error, hasCause(hasMessage(containsString(expectedMessage))));
  }
}
