/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.message;

import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_REACTIVE;

import static java.nio.ByteBuffer.wrap;
import static java.nio.charset.StandardCharsets.UTF_8;

import static org.apache.commons.io.IOUtils.toByteArray;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.anEmptyMap;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.junit.internal.matchers.ThrowableCauseMatcher.hasCause;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;
import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.api.util.MultiMap;
import org.mule.service.http.netty.impl.message.content.NettyFeedableHttpEntity;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Story;
import org.junit.jupiter.api.Test;

@Story(HTTP_2_REACTIVE)
public class FeedableHttpEntityTestCase extends AbstractHttpTestCase {

  public static final String HELLO = "Hello ";
  public static final String WORLD = "world!";
  public static final String HELLO_WORLD = HELLO + WORLD;

  @Test
  void lengthDependsOnParameter() {
    var entityKnownLength = new NettyFeedableHttpEntity(10L);
    assertThat(entityKnownLength.getBytesLength().isPresent(), is(true));
    assertThat(entityKnownLength.getBytesLength().getAsLong(), is(10L));

    var entityUnknownLength = new NettyFeedableHttpEntity();
    assertThat(entityUnknownLength.getBytesLength().isPresent(), is(false));

    var entityLengthMinusOne = new NettyFeedableHttpEntity(-1L);
    assertThat(entityLengthMinusOne.getBytesLength().isPresent(), is(false));
  }

  @Test
  void contentCanBeConsumedAsStream() throws IOException {
    var originalBytes = "Hello test!".getBytes();

    var entity = new NettyFeedableHttpEntity();
    entity.feed(wrap(originalBytes));
    entity.complete();

    var readBytes = toByteArray(entity.getContent());
    assertThat(readBytes, is(originalBytes));
  }

  @Test
  void contentCanBeConsumedAsReactive() throws IOException, ExecutionException, InterruptedException {
    var originalString = "Hello test!";

    var entity = new NettyFeedableHttpEntity();
    entity.feed(wrap(originalString.getBytes()));
    entity.complete();

    var stringBuilder = new StringBuilder();
    var aggregatedPayload = new CompletableFuture<String>();
    entity.onData(data -> stringBuilder.append(UTF_8.decode(data)));
    entity.onComplete((ts, err) -> aggregatedPayload.complete(stringBuilder.toString()));

    assertThat(aggregatedPayload.get(), is(originalString));
  }

  @Test
  void whenBeingConsumedAsStream_canNotBeConsumedAsReactive() {
    var entity = new NettyFeedableHttpEntity();

    // Start consuming it as an InputStream...
    entity.getContent();

    var error = assertThrows(IllegalStateException.class, () -> entity.onData(data -> {
    }));
    assertThat(error.getMessage(), is("This entity is being consumed as an input stream"));
  }

  @Test
  void whenBeingConsumedAsReactive_canNotBeConsumedAsStream() {
    var entity = new NettyFeedableHttpEntity();

    // Start consuming it reactively...
    entity.onData(data -> {
    });

    var error = assertThrows(IllegalStateException.class, entity::getContent);
    assertThat(error.getMessage(), is("This entity is being consumed reactively"));
  }

  @Test
  void cannotRegisterTwoReactiveDataConsumers() {
    var entity = new NettyFeedableHttpEntity();

    // Start consuming it reactively...
    entity.onData(data -> {
    });

    var error = assertThrows(IllegalStateException.class, () -> entity.onData(data -> {
    }));
    assertThat(error.getMessage(), is("A consumer has already been registered for this entity"));
  }

  @Test
  void feedContentAndFinish() throws Exception {
    var feedableEntity = new NettyFeedableHttpEntity();
    feedableEntity.feed(wrap(HELLO.getBytes()));
    feedableEntity.feed(wrap(WORLD.getBytes()));
    feedableEntity.complete();

    String out = new String(feedableEntity.getBytes());
    assertThat(out, is(HELLO_WORLD));

    var futureTrailers = new CompletableFuture<MultiMap<String, String>>();
    feedableEntity.onComplete((trailers, throwable) -> {
      if (throwable != null) {
        futureTrailers.completeExceptionally(throwable);
      } else {
        futureTrailers.complete(trailers);
      }
    });
    var trailers = futureTrailers.get();
    assertThat(trailers.isEmpty(), is(true));
    assertThat(trailers, is(anEmptyMap()));
  }

  @Test
  void tryFeedingAfterCompletionFails() {
    var feedableEntity = new NettyFeedableHttpEntity();

    feedableEntity.complete();
    var error = assertThrows(IOException.class, () -> feedableEntity.feed(wrap(HELLO.getBytes())));

    assertThat(error, hasMessage(containsString("This entity is already completed")));
  }

  @Test
  void tryFeedingAfterCompletionWithTrailersFails() {
    var feedableEntity = new NettyFeedableHttpEntity();

    var trailers = new MultiMap.StringMultiMap();
    trailers.put("status", "blah");
    feedableEntity.completeWithTrailers(trailers);
    var error = assertThrows(IOException.class, () -> feedableEntity.feed(wrap(HELLO.getBytes())));

    assertThat(error, hasMessage(containsString("This entity is already completed")));
  }

  @Test
  void tryFeedingAfterErrorFails() {
    var feedableEntity = new NettyFeedableHttpEntity();

    var userError = new NullPointerException("EXPECTED");
    feedableEntity.error(userError);
    var error = assertThrows(NullPointerException.class, () -> feedableEntity.feed(wrap(HELLO.getBytes())));

    assertThat(error, hasMessage(containsString("EXPECTED")));
  }

  @Test
  void errorAfterCompleteFails() {
    var feedableEntity = new NettyFeedableHttpEntity();

    var userError = new NullPointerException("ERROR AFTER COMPLETE");
    feedableEntity.complete();
    var error = assertThrows(IllegalStateException.class, () -> feedableEntity.error(userError));

    assertThat(error, hasMessage(containsString("This entity is already completed")));
  }

  @Test
  void secondErrorInvocationFails() {
    var feedableEntity = new NettyFeedableHttpEntity();

    var firstError = new NullPointerException("USER ERROR");
    feedableEntity.error(firstError);

    var secondError = new NullPointerException("ILLEGAL ERROR");
    var invocationError = assertThrows(IllegalStateException.class, () -> feedableEntity.error(secondError));

    assertThat(invocationError, hasMessage(containsString("This entity has already been marked with an error")));
    assertThat(invocationError, hasCause(hasMessage(containsString("USER ERROR"))));
  }

  @Test
  void secondCompleteFails() {
    var feedableEntity = new NettyFeedableHttpEntity();

    feedableEntity.complete();
    var error = assertThrows(IllegalStateException.class, feedableEntity::complete);

    assertThat(error, hasMessage(containsString("This entity is already completed")));
  }
}
