/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message.content;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;

import static org.apache.commons.io.IOUtils.toByteArray;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.entity.FeedableHttpEntity;
import org.mule.runtime.http.api.domain.entity.EntitySubscription;
import org.mule.service.http.netty.impl.streaming.BlockingBidirectionalStream;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import org.slf4j.Logger;
import reactor.core.publisher.Sinks;

public class NettyFeedableHttpEntity implements FeedableHttpEntity {

  public static final long UNKNOWN_CONTENT_LENGTH = -1L;

  private static final Logger LOGGER = getLogger(NettyFeedableHttpEntity.class);

  private final long contentLength;
  private final CompletableFuture<MultiMap<String, String>> futureTrailers;
  private final Sinks.Many<ByteBuffer> dataSink;

  private final Object consumptionLock = new Object();
  private ConsumptionMode consumptionMode;
  private InputStream asInputStream;

  private final Object completionLock = new Object();
  private boolean isCompleted;
  private boolean closedFromReader = false;
  private Exception savedError;

  public NettyFeedableHttpEntity() {
    this(UNKNOWN_CONTENT_LENGTH);
  }

  public NettyFeedableHttpEntity(long contentLength) {
    this.contentLength = contentLength;
    this.consumptionMode = ConsumptionMode.NOT_YET_CONSUMED;
    this.futureTrailers = new CompletableFuture<>();
    this.dataSink = Sinks.many().unicast().onBackpressureBuffer();
  }

  @Override
  public boolean isStreaming() {
    return consumptionMode.allowsStreaming;
  }

  @Override
  public boolean isReactive() {
    return consumptionMode.allowsReactive;
  }

  @Override
  public InputStream getContent() {
    synchronized (consumptionLock) {
      return switch (consumptionMode) {
        case NOT_YET_CONSUMED, AS_INPUT_STREAM -> getOrCreateInputStream();
        case AS_REACTIVE -> throw new IllegalStateException("This entity is being consumed reactively");
      };
    }
  }

  private InputStream getOrCreateInputStream() {
    if (asInputStream == null) {
      consumptionMode = ConsumptionMode.AS_INPUT_STREAM;
      var bidirectionalStream = new BlockingBidirectionalStream();
      dataSink.asFlux()
          .doOnNext(data -> {
            byte[] bytes = new byte[data.remaining()];
            data.get(bytes);

            LOGGER.debug("Feeding piped entity with {} bytes", bytes.length);
            try {
              bidirectionalStream.write(bytes, 0, bytes.length);
            } catch (IOException e) {
              throw new RuntimeException(e);
            }
          })
          .doOnError(error -> {
            LOGGER.debug("Canceling piped entity", error);
            futureTrailers.completeExceptionally(error);
            bidirectionalStream.cancel(error);
          })
          .doOnComplete(() -> {
            LOGGER.debug("Completing piped entity successfully");
            bidirectionalStream.close();
          })
          .subscribe();

      asInputStream = bidirectionalStream.getInputStream();
    }
    return asInputStream;
  }

  @Override
  public byte[] getBytes() throws IOException {
    return toByteArray(getContent());
  }

  @Override
  public Optional<Long> getLength() {
    return contentLength == UNKNOWN_CONTENT_LENGTH ? Optional.empty() : Optional.of(contentLength);
  }

  @Override
  public OptionalLong getBytesLength() {
    return contentLength == UNKNOWN_CONTENT_LENGTH ? OptionalLong.empty() : OptionalLong.of(contentLength);
  }

  @Override
  public void feed(ByteBuffer data) throws IOException {
    synchronized (completionLock) {
      checkIsOpen();
      LOGGER.debug("Feeding entity with {} bytes", data.remaining());
      var res = dataSink.tryEmitNext(data);
      LOGGER.debug("Feed result: {}", res);
    }
  }

  private void checkIsOpen() throws IOException {
    if (closedFromReader) {
      throw new IllegalStateException("HTTP Entity was closed from reader side");
    }

    if (isCompleted) {
      throw new IOException("This entity is already completed");
    }

    if (savedError instanceof IOException ioException) {
      throw ioException;
    }

    if (savedError instanceof RuntimeException runtimeException) {
      throw runtimeException;
    }

    if (savedError != null) {
      throw new IOException(savedError);
    }
  }

  @Override
  public void error(Exception error) {
    synchronized (completionLock) {
      checkNotCompleted();

      savedError = error;
      LOGGER.debug("Marking entity with error", error);
      var res = dataSink.tryEmitError(error);
      LOGGER.debug("Emit error result: {}", res);
    }
  }

  private void checkNotCompleted() {
    if (isCompleted) {
      throw new IllegalStateException("This entity is already completed");
    }

    if (savedError != null) {
      throw new IllegalStateException("This entity has already been marked with an error", savedError);
    }
  }

  @Override
  public void complete() {
    completeWithTrailers(emptyMultiMap());
  }

  @Override
  public void completeWithTrailers(MultiMap<String, String> trailers) {
    synchronized (completionLock) {
      checkNotCompleted();

      isCompleted = true;
      LOGGER.atDebug().log(() -> "Completing HTTP entity with " + trailers.size() + " trailers");
      futureTrailers.complete(trailers);
      var res = dataSink.tryEmitComplete();
      LOGGER.debug("Emit complete result: {}", res);
    }
  }

  @Override
  public void onComplete(BiConsumer<? super MultiMap<String, String>, ? super Throwable> completionCallback) {
    LOGGER.debug("Registering a callback for the entity completion");
    futureTrailers.whenComplete(completionCallback);
  }

  @Override
  public EntitySubscription onData(Consumer<ByteBuffer> consumer) {
    synchronized (consumptionLock) {
      switch (consumptionMode) {
        case NOT_YET_CONSUMED -> {
          return registerDataConsumer(consumer);
        }
        case AS_REACTIVE -> throw new IllegalStateException("A consumer has already been registered for this entity");
        case AS_INPUT_STREAM -> throw new IllegalStateException("This entity is being consumed as an input stream");
        default -> throw new IllegalStateException("Unknown consumption mode " + consumptionMode);
      }
    }
  }

  private EntitySubscription registerDataConsumer(Consumer<ByteBuffer> consumer) {
    consumptionMode = ConsumptionMode.AS_REACTIVE;
    var disposable = dataSink.asFlux().doOnNext(consumer).subscribe();
    return () -> {
      synchronized (completionLock) {
        disposable.dispose();
        this.closedFromReader = true;
      }
    };
  }

  private enum ConsumptionMode {

    AS_INPUT_STREAM(true, false),

    AS_REACTIVE(false, true),

    NOT_YET_CONSUMED(true, true);

    public final boolean allowsStreaming;

    public final boolean allowsReactive;

    ConsumptionMode(boolean allowsStreaming, boolean allowsReactive) {
      this.allowsStreaming = allowsStreaming;
      this.allowsReactive = allowsReactive;
    }
  }
}
