/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.streaming;

import static org.mule.service.http.netty.impl.server.ChannelPromises.promiseToCallback;

import static io.netty.buffer.Unpooled.EMPTY_BUFFER;
import static io.netty.buffer.Unpooled.wrappedBuffer;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.service.http.netty.impl.server.HttpWriter;

import java.io.IOException;

import io.netty.channel.ChannelHandlerContext;
import org.slf4j.Logger;

/**
 * Writes a {@link HttpEntity} to a {@link ChannelHandlerContext} in a reactive fashion, using the reactive methods. Only usable
 * for entities returning {@code true} for {@link HttpEntity#isReactive()}.
 */
public class ReactiveEntitySender {

  private static final Logger LOGGER = getLogger(ReactiveEntitySender.class);

  private final ChannelHandlerContext ctx;
  private final Runnable beforeWrite;
  private final StatusCallback statusCallback;
  private final HttpWriter writer;

  private final HttpEntity entity;

  public ReactiveEntitySender(HttpEntity entity, ChannelHandlerContext ctx, Runnable beforeWrite,
                              StatusCallback statusCallback, HttpWriter writer) {
    if (!entity.isReactive()) {
      throw new IllegalArgumentException("HTTP Entity does not allow reactive consumption");
    }

    this.ctx = ctx;
    this.writer = writer;
    this.beforeWrite = beforeWrite;
    this.statusCallback = statusCallback;
    this.entity = entity;
  }

  public void start() throws IOException {
    entity.onData(buffer -> {
      beforeWrite.run();
      LOGGER.debug("About to send entity data");
      writer.writeContent(wrappedBuffer(buffer), false, promiseToCallback(ctx, statusCallback, false));
    });

    entity.onComplete((trailers, throwable) -> {
      beforeWrite.run();
      LOGGER.debug("About to send {} trailers", trailers.size());
      if (trailers.isEmpty()) {
        writer.writeContent(EMPTY_BUFFER, true, promiseToCallback(ctx, statusCallback, true));
      } else {
        writer.writeTrailers(trailers, promiseToCallback(ctx, statusCallback, true));
      }
    });
  }
}
