/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.common.client.sse;

import static java.lang.Long.parseLong;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.sse.ServerSentEvent;
import org.mule.runtime.http.api.sse.client.SseListener;
import org.mule.service.http.common.message.sse.SseEventBuilder;

import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;

public class ServerSentEventDecoder implements AutoCloseable {

  private static final Logger LOGGER = getLogger(ServerSentEventDecoder.class);

  // Java8 regex equivalent to "\\r?\\n|\\r" (essentially matches any combination of \r and \n).
  private static final String LINE_BREAK_REGEX = "\\R";

  private final SseEventBuilder eventBuilder = new SseEventBuilder();
  private final SseListener eventListener;
  private static final AtomicLong idGenerator = new AtomicLong();

  private String notParsedData = "";
  private final Long id;

  public ServerSentEventDecoder(SseListener eventListener) {
    this.eventListener = eventListener;
    this.id = idGenerator.incrementAndGet();
  }

  public void decode(byte[] data, int actualRead) {
    notParsedData += new String(data, 0, actualRead);
    String[] lines = notParsedData.split(LINE_BREAK_REGEX, -1);
    int linesNumber = lines.length;

    for (int i = 0; i < linesNumber - 1; i++) {
      LOGGER.trace("SSE Decoder ({}): Handling line: [{}]", id, lines[i]);
      handleLine(lines[i]);
    }

    // the last element is the string after the last line separator
    notParsedData = lines[linesNumber - 1];
    LOGGER.trace("SSE Decoder ({}): Found data after last line separator: [{}]", id, notParsedData);
  }

  public Long getId() {
    return id;
  }

  @Override
  public void close() {
    LOGGER.trace("SSE Decoder ({}): Finished parsing the stream. Closing...", id);
    eventListener.onClose();
  }

  private void handleLine(String line) {
    if (null == line || line.isEmpty()) {
      ServerSentEvent serverSentEvent = eventBuilder.buildAndClear();
      LOGGER.debug("SSE Decoder ({}): Reading server-sent event: [{}]", id, serverSentEvent);
      this.eventListener.onEvent(serverSentEvent);
    } else if (line.startsWith("data:")) {
      eventBuilder.withData(line.substring(5).trim());
    } else if (line.startsWith("event:")) {
      eventBuilder.withName(line.substring(6).trim());
    } else if (line.startsWith("id:")) {
      eventBuilder.withId(line.substring(3).trim());
    } else if (line.startsWith("retry:")) {
      eventBuilder.withRetryDelay(parseRetryDelay(line.substring(6).trim()));
    }
  }

  // Spec states that if the retry delay is not a number, we should just ignore it.
  private Long parseRetryDelay(String asString) {
    try {
      return parseLong(asString);
    } catch (NumberFormatException e) {
      LOGGER.debug("SSE Decoder ({}): Failed to parse retry delay because '{}' is not a number", id, asString, e);
      return null;
    }
  }
}
