/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static org.mule.service.http.netty.impl.client.ReactorNettyClient.TRAILERS_FUTURE;

import static org.slf4j.LoggerFactory.getLogger;

import java.util.List;

import io.netty.channel.ChannelHandler.Sharable;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.MessageToMessageEncoder;
import io.netty.handler.codec.http2.DefaultHttp2DataFrame;
import io.netty.handler.codec.http2.DefaultHttp2Headers;
import io.netty.handler.codec.http2.DefaultHttp2HeadersFrame;
import io.netty.handler.codec.http2.Http2DataFrame;
import io.netty.handler.codec.http2.Http2Frame;
import io.netty.util.ReferenceCounted;
import org.slf4j.Logger;

@Sharable
public class TrailersSenderHandler extends MessageToMessageEncoder<Http2Frame> {

  private static final Logger LOGGER = getLogger(TrailersSenderHandler.class);

  @Override
  protected void encode(ChannelHandlerContext ctx, Http2Frame frame, List<Object> outList) throws Exception {
    LOGGER.trace("TrailersSenderHandler#write() called with object of type: {}",
                 frame != null ? frame.getClass().getName() : "null");

    if (frame instanceof ReferenceCounted rc) {
      rc.retain();
    }

    if (frame instanceof Http2DataFrame dataFrame && dataFrame.isEndStream()) {
      var trailersMap = ctx.channel().attr(TRAILERS_FUTURE).get().get();

      if (trailersMap.isEmpty()) {
        outList.add(frame);
        return;
      }

      var stream = dataFrame.stream();

      var lastData = new DefaultHttp2DataFrame(dataFrame.content(), false, dataFrame.padding()).stream(stream);
      outList.add(lastData);

      var trailers = new DefaultHttp2Headers(false);
      trailersMap.forEach(trailers::add);
      var trailersFrame = new DefaultHttp2HeadersFrame(trailers, true).stream(stream);
      outList.add(trailersFrame);

      return;
    }

    outList.add(frame);
  }
}
